<?php

namespace boru\ocr\Layout\Strategy;

use boru\ocr\Layout\Support\BoundsCalculator;
use boru\ocr\Layout\Support\LayoutDiagnostics;

/**
 * Chooses best ordering for document-like pages:
 * - banded
 * - legacy (optional multi-column)
 * - mixed (single-column top + columnar bottom)
 *
 * Uses simple penalties to avoid header drift.
 */
class AutoDocumentStrategy implements ReadingOrderStrategyInterface
{
    /** @var BoundsCalculator */
    protected $bounds;

    /** @var BandedOrderStrategy */
    protected $banded;

    /** @var LegacyOrderStrategy */
    protected $legacy;

    /** @var int */
    protected $bandTolerancePx = 14;

    /** @var bool */
    protected $detectMixedLayout = true;

    public function __construct(BoundsCalculator $bounds, BandedOrderStrategy $banded, LegacyOrderStrategy $legacy, array $options = array())
    {
        $this->bounds = $bounds;
        $this->banded = $banded;
        $this->legacy = $legacy;

        if (isset($options['bandTolerancePx'])) $this->bandTolerancePx = (int)$options['bandTolerancePx'];
        if ($this->bandTolerancePx < 4) $this->bandTolerancePx = 4;

        if (array_key_exists('detectMixedLayout', $options)) $this->detectMixedLayout = (bool)$options['detectMixedLayout'];
    }

    public function order(array $lines, LayoutDiagnostics $diag = null)
    {
        $banded = $this->banded->order($lines, null);
        $pBanded = $this->penalty($banded);

        $legacy = $this->legacy->order($lines, null);
        $pLegacy = $this->penalty($legacy);

        $chosen = $banded;
        $chosenName = 'banded';
        $best = $pBanded;

        if ($pLegacy < $best) {
            $best = $pLegacy;
            $chosen = $legacy;
            $chosenName = 'legacy';
        }

        $mixed = null;
        $pMixed = null;
        $mixedInfo = null;

        if ($this->detectMixedLayout) {
            $mixedInfo = $this->detectMixedInfo($lines);
            if ($mixedInfo['mixed']) {
                $mixedStrat = new MixedLayoutStrategy(
                    $this->bounds,
                    $this->banded,
                    $this->legacy,
                    array('breakY' => $mixedInfo['breakY'], 'headerBandPx' => $mixedInfo['headerBandPx'])
                );
                $mixed = $mixedStrat->order($lines, null);
                $pMixed = $this->penalty($mixed);

                if ($pMixed !== null && $pMixed <= $best) {
                    $best = $pMixed;
                    $chosen = $mixed;
                    $chosenName = 'mixed';
                }
            }
        }

        if ($diag) {
            $diag->strategy = $chosenName;
            $diag->metrics['bandedPenalty'] = $pBanded;
            $diag->metrics['legacyPenalty'] = $pLegacy;
            if ($mixedInfo) {
                $diag->metrics['mixedDetected'] = (bool)$mixedInfo['mixed'];
                $diag->metrics['mixedBreakY'] = (int)$mixedInfo['breakY'];
                $diag->metrics['mixedTopColumnar'] = (bool)$mixedInfo['topColumnar'];
                $diag->metrics['mixedBottomColumnar'] = (bool)$mixedInfo['bottomColumnar'];
                if ($pMixed !== null) $diag->metrics['mixedPenalty'] = $pMixed;
            }
        }

        return $chosen;
    }

    protected function penalty(array $orderedLines)
    {
        $tops = array();
        foreach ($orderedLines as $ln) {
            $b = $this->bounds->lineBounds($ln);
            if (!$b) continue;
            $tops[] = (int)$b['minTop'];
        }
        if (count($tops) < 3) return 0;

        $tol = (int)max(8, (int)floor($this->bandTolerancePx * 0.75));

        $inversions = 0;
        $invMagnitude = 0;

        for ($i = 1; $i < count($tops); $i++) {
            $delta = $tops[$i - 1] - $tops[$i];
            if ($delta > $tol) {
                $inversions++;
                $invMagnitude += $delta;
            }
        }

        $earlyN = min(12, count($tops) - 1);
        $earlyMax = $tops[0];
        for ($i = 1; $i <= $earlyN; $i++) {
            if ($tops[$i] > $earlyMax) $earlyMax = $tops[$i];
        }
        $laterMin = $tops[$earlyN];
        for ($i = $earlyN; $i < count($tops); $i++) {
            if ($tops[$i] < $laterMin) $laterMin = $tops[$i];
        }
        $earlyJumpPenalty = 0;
        if ($laterMin + $tol < $earlyMax) {
            $earlyJumpPenalty = (int)(($earlyMax - $laterMin) / 5);
        }

        return ($inversions * 200) + (int)floor($invMagnitude / 3) + ($earlyJumpPenalty * 10);
    }

    protected function detectMixedInfo(array $lines)
    {
        $out = array(
            'mixed' => false,
            'breakY' => 0,
            'topColumnar' => false,
            'bottomColumnar' => false,
            'headerBandPx' => 300,
        );

        // Compute page vertical span
        $minTop = null;
        $maxBottom = null;

        foreach ($lines as $ln) {
            $b = $this->bounds->lineBounds($ln);
            if (!$b) continue;
            $t = (int)$b['minTop'];
            $bot = (int)$b['maxBottom'];
            if ($minTop === null || $t < $minTop) $minTop = $t;
            if ($maxBottom === null || $bot > $maxBottom) $maxBottom = $bot;
        }
        if ($minTop === null || $maxBottom === null) return $out;

        $height = $maxBottom - $minTop;
        if ($height < 250) return $out;

        $topMax = $minTop + (int)floor($height * 0.35);
        $bottomMin = $minTop + (int)floor($height * 0.65);

        $topLines = array();
        $bottomLines = array();
        foreach ($lines as $ln) {
            $b = $this->bounds->lineBounds($ln);
            if (!$b) continue;
            if ((int)$b['minTop'] <= $topMax) $topLines[] = $ln;
            if ((int)$b['minTop'] >= $bottomMin) $bottomLines[] = $ln;
        }

        $topCol = $this->sliceIsColumnar($topLines);
        $bottomCol = $this->sliceIsColumnar($bottomLines);

        $out['topColumnar'] = $topCol;
        $out['bottomColumnar'] = $bottomCol;
        $out['breakY'] = $bottomMin;
        $out['mixed'] = (!$topCol && $bottomCol);

        return $out;
    }

    protected function sliceIsColumnar(array $lines)
    {
        if (count($lines) < 6) return false;

        $mids = array();
        $minX = null;
        $maxX = null;

        foreach ($lines as $ln) {
            $b = $this->bounds->lineBounds($ln);
            if (!$b) continue;
            $mids[] = (int)$b['midX'];
            if ($minX === null || $b['minLeft'] < $minX) $minX = $b['minLeft'];
            if ($maxX === null || $b['maxRight'] > $maxX) $maxX = $b['maxRight'];
        }

        if (count($mids) < 6 || $minX === null || $maxX === null) return false;

        $width = (int)($maxX - $minX);
        if ($width < 260) return false;

        sort($mids);

        $maxGap = 0;
        $maxGapIdx = -1;

        for ($i = 1; $i < count($mids); $i++) {
            $gap = (int)($mids[$i] - $mids[$i - 1]);
            if ($gap > $maxGap) {
                $maxGap = $gap;
                $maxGapIdx = $i;
            }
        }

        $gapThresh = (int)max(60, (int)floor($width * 0.18));

        if ($maxGapIdx <= 1) return false;
        if ($maxGapIdx >= count($mids) - 2) return false;

        return ($maxGap >= $gapThresh);
    }
}
