<?php

namespace boru\ocr\Layout\Strategy;

use boru\ocr\Layout\Support\LayoutDiagnostics;
use boru\ocr\Layout\Support\BoundsCalculator;
use boru\ocr\Tesseract\Tsv\TsvRow;

/**
 * Table-first ordering.
 *
 * - Uses banded (row) ordering to preserve row sequence.
 * - Within each band, left->right.
 * - Slightly tighter band tolerance than generic docs (tables align rows tightly).
 *
 * NOTE: Rendering (spacing/column hints) should be enabled via LineRenderer options
 * (e.g., preserveGapSpacing) when this strategy is selected.
 */
class TableLayoutStrategy implements ReadingOrderStrategyInterface
{
    /** @var BoundsCalculator */
    protected $bounds;

    /** @var int */
    protected $bandTolerancePx = 10;

    public function __construct(BoundsCalculator $bounds = null, array $options = array())
    {
        if (isset($options['bandTolerancePx'])) $this->bandTolerancePx = (int)$options['bandTolerancePx'];
        if ($this->bandTolerancePx < 4) $this->bandTolerancePx = 4;

        $this->bounds = $bounds ?: new BoundsCalculator();
    }

    /**
     * @param array<int,array> $lines
     * @param LayoutDiagnostics|null $diag
     * @return array<int,array>
     */
    public function order(array $lines, LayoutDiagnostics $diag = null)
    {
        // Sort words within each line
        foreach ($lines as &$line) {
            usort($line, array(__CLASS__, 'cmpLeft'));
        }
        unset($line);

        // Build meta for banding
        $meta = array();
        $count = count($lines);
        for ($i = 0; $i < $count; $i++) {
            $b = $this->bounds->lineBounds($lines[$i]);
            if ($b === null) continue;
            $meta[] = array(
                'idx' => $i,
                'top' => (int)$b['top'],
                'left' => (int)$b['left'],
            );
        }

        if (count($meta) < 2) {
            // fallback: just sort by top then left
            usort($lines, array(__CLASS__, 'cmpLineTopLeft'));
            if ($diag) {
                $diag->strategy = 'table';
                $diag->profile = $diag->profile ? $diag->profile : 'table';
                $diag->notes[] = 'TableLayoutStrategy: fallback top/left sort (insufficient bounds).';
            }
            return $lines;
        }

        usort($meta, array(__CLASS__, 'cmpMetaTopLeft'));

        // Group into row bands
        $bands = array();
        $current = array();
        $bandTop = null;
        $tol = (int)$this->bandTolerancePx;

        foreach ($meta as $m) {
            if ($bandTop === null) {
                $bandTop = $m['top'];
                $current[] = $m;
                continue;
            }
            if (abs($m['top'] - $bandTop) <= $tol) {
                $current[] = $m;
            } else {
                $bands[] = $current;
                $current = array($m);
                $bandTop = $m['top'];
            }
        }
        if (!empty($current)) $bands[] = $current;

        // Within a band, sort by left
        $ordered = array();
        foreach ($bands as $band) {
            usort($band, array(__CLASS__, 'cmpMetaLeft'));
            foreach ($band as $m) {
                $ordered[] = $lines[$m['idx']];
            }
        }

        if ($diag) {
            $diag->strategy = 'table';
            $diag->profile = 'table';
        }

        return $ordered;
    }

    public static function cmpLeft(TsvRow $a, TsvRow $b)
    {
        if ($a->left === $b->left) {
            if ($a->top === $b->top) return 0;
            return ($a->top < $b->top) ? -1 : 1;
        }
        return ($a->left < $b->left) ? -1 : 1;
    }

    public static function cmpLineTopLeft($lineA, $lineB)
    {
        // crude fallback: compare first rows
        if (empty($lineA) && empty($lineB)) return 0;
        if (empty($lineA)) return 1;
        if (empty($lineB)) return -1;

        /** @var TsvRow $a */
        $a = $lineA[0];
        /** @var TsvRow $b */
        $b = $lineB[0];

        if ($a->top === $b->top) {
            if ($a->left === $b->left) return 0;
            return ($a->left < $b->left) ? -1 : 1;
        }
        return ($a->top < $b->top) ? -1 : 1;
    }

    public static function cmpMetaTopLeft($a, $b)
    {
        if ($a['top'] === $b['top']) {
            if ($a['left'] === $b['left']) return 0;
            return ($a['left'] < $b['left']) ? -1 : 1;
        }
        return ($a['top'] < $b['top']) ? -1 : 1;
    }

    public static function cmpMetaLeft($a, $b)
    {
        if ($a['left'] === $b['left']) return 0;
        return ($a['left'] < $b['left']) ? -1 : 1;
    }
}
