<?php

namespace boru\ocr\Layout\Strategy;

use boru\ocr\Layout\Support\BoundsCalculator;
use boru\ocr\Layout\Support\LayoutDiagnostics;
use boru\ocr\Layout\Support\LineRenderer;
use boru\ocr\Layout\Support\TableLineRenderer;
use boru\ocr\Layout\Support\TableRegionDetector;
use boru\ocr\Layout\LayoutOptions;
use boru\ocr\Tesseract\Tsv\TsvRow;

/**
 * Diagram layout:
 * - cluster words into spatial regions
 * - detect title block (dense cluster near bottom/right)
 * - render as region blocks, preserving local ordering
 */
class DiagramLayoutStrategy
{
    /** @var BoundsCalculator */
    protected $bounds;

    /** @var BandedOrderStrategy */
    protected $banded;

    /** @var LineRenderer */
    protected $renderer;

    /** @var int px */
    protected $clusterPadPx = 70;

    /** @var int */
    protected $minRegionWords = 8;

    /** @var TableRegionDetector */
    protected $tableRegionDetector;

    /** @var TableLineRenderer */
    protected $tableRenderer;

    /** @var bool */
    protected $detectDiagramTableRegions = true;

    /** @var bool */
    protected $diagramTableRegionMarkers = false;

    /** @var LayoutOptions */
    protected $layoutOptions = null;

    public function __construct(BoundsCalculator $bounds, BandedOrderStrategy $banded, LineRenderer $renderer, LayoutOptions $options = null)
    {
        $this->layoutOptions = LayoutOptions::create($options);
        $this->bounds = $bounds;
        $this->banded = $banded;
        $this->renderer = $renderer;

        $this->clusterPadPx = $this->layoutOptions->clusterPadPx;
        $this->minRegionWords = $this->layoutOptions->minRegionWords;

        $this->detectDiagramTableRegions = $this->layoutOptions->detectTableRegions;
        $this->diagramTableRegionMarkers = $this->layoutOptions->tableRegionMarkers;

        $this->tableRegionDetector = new TableRegionDetector($this->bounds, $this->layoutOptions);
        $this->tableRenderer = new TableLineRenderer($this->layoutOptions);
    }

    /**
     * Render diagram baseline text.
     *
     * @param TsvRow[] $wordRows
     * @param array<int, TsvRow[]> $lines
     * @param LayoutDiagnostics|null $diag
     * @return string
     */
    public function buildText(array $wordRows, array $lines, LayoutDiagnostics $diag = null)
    {
        $regions = $this->clusterWords($wordRows);

        // fallback if clustering didn't produce meaningful regions
        if (count($regions) <= 1) {
            $ordered = $this->banded->order($lines, null);
            $out = array();
            foreach ($ordered as $ln) {
                $t = trim($this->renderer->renderLine($ln));
                if ($t !== '') $out[] = $t;
            }
            if ($diag) {
                $diag->strategy = 'diagram:fallback-banded';
                $diag->metrics['regionCount'] = count($regions);
            }
            return implode("\n", $out);
        }

        // detect title block region
        $titleIdx = $this->detectTitleBlock($regions);

        // order regions by centroid top->bottom, left->right, but keep title block last
        $orderedRegions = $this->orderRegions($regions, $titleIdx);

        $out = array();
        $i = 1;

        foreach ($orderedRegions as $ri) {
            $r = $regions[$ri];
            $label = ($ri === $titleIdx) ? 'TITLE BLOCK' : ('REGION ' . $i);
            if ($ri !== $titleIdx) $i++;

            $out[] = '[' . $label . ' @ x=' . $r['minLeft'] . ',y=' . $r['minTop'] . ',w=' . ($r['maxRight'] - $r['minLeft']) . ',h=' . ($r['maxBottom'] - $r['minTop']) . ']';

            // Build lines for region: filter original lines whose bounds intersect region
            $regionLines = $this->linesInRegion($lines, $r);

            $regionLines = $this->banded->order($regionLines, null);

            // Optionally lift table-like sub-regions inside this diagram region (BOM/title blocks/etc.)
            if ($this->detectDiagramTableRegions) {
                $regions2 = $this->tableRegionDetector->detect($regionLines, 'diagram');

                if ($diag) {
                    if (!isset($diag->metrics['tableCandidates'])) $diag->metrics['tableCandidates'] = array();
                    foreach ($regions2 as $rg) {
                        $cand = $rg;
                        $cand['kind'] = 'region';
                        $cand['profile'] = 'diagram';
                        $cand['region'] = $label;
                        $diag->metrics['tableCandidates'][] = $cand;
                    }
                }

                // Build lookup line->region2
                $lineToR2 = array();
                $r2id = 0;
                foreach ($regions2 as $rg) {
                    for ($k = (int)$rg['start']; $k <= (int)$rg['end']; $k++) {
                        $lineToR2[$k] = $r2id;
                    }
                    $r2id++;
                }

                $open = null;
                for ($k = 0; $k < count($regionLines); $k++) {
                    $ln = $regionLines[$k];
                    $inTable = array_key_exists($k, $lineToR2);
                    $thisRid = $inTable ? $lineToR2[$k] : null;

                    if ($this->diagramTableRegionMarkers) {
                        if ($inTable && $open === null) {
                            $out[] = '--- table ---';
                            $open = $thisRid;
                        } elseif (!$inTable && $open !== null) {
                            $out[] = '--- /table ---';
                            $open = null;
                        } elseif ($inTable && $open !== null && $open !== $thisRid) {
                            $out[] = '--- /table ---';
                            $out[] = '--- table ---';
                            $open = $thisRid;
                        }
                    }

                    $t = $inTable
                        ? trim($this->tableRenderer->render($ln))
                        : trim($this->renderer->renderLine($ln));

                    if ($t !== '') $out[] = $t;
                }
                if ($this->diagramTableRegionMarkers && $open !== null) {
                    $out[] = '--- /table ---';
                }
            } else {
                foreach ($regionLines as $ln) {
                    $t = trim($this->renderer->renderLine($ln));
                    if ($t === '') continue;
                    $out[] = $t;
                }
            }

            $out[] = '';
        }

        if ($diag) {
            $diag->strategy = 'diagram:regions';
            $diag->metrics['regionCount'] = count($regions);
            $diag->metrics['titleBlockDetected'] = ($titleIdx !== null);
        }

        return rtrim(implode("\n", $out));
    }

    /**
     * Cluster word rows into region boxes using greedy bbox merging.
     *
     * @param TsvRow[] $rows
     * @return array<int,array{minLeft:int,maxRight:int,minTop:int,maxBottom:int,words:array}>
     */
    protected function clusterWords(array $rows)
    {
        $words = array();
        foreach ($rows as $r) {
            if ((int)$r->level !== 5) continue;
            if ($r->text === '') continue;
            if ($r->conf < 0) continue;
            $words[] = $r;
        }
        if (count($words) === 0) return array();

        // Sort by top then left for stable greedy merging
        usort($words, array(__CLASS__, 'cmpWordTopLeft'));

        $regions = array();

        foreach ($words as $w) {
            $wBox = array(
                'minLeft' => (int)$w->left,
                'maxRight' => (int)$w->right(),
                'minTop' => (int)$w->top,
                'maxBottom' => (int)$w->top + (int)$w->height,
                'words' => array($w),
            );

            $mergedInto = null;

            // Try to merge into an existing region if close
            for ($i = 0; $i < count($regions); $i++) {
                if ($this->boxesNear($regions[$i], $wBox, $this->clusterPadPx)) {
                    $regions[$i] = $this->mergeBoxes($regions[$i], $wBox);
                    $mergedInto = $i;
                    break;
                }
            }

            if ($mergedInto === null) {
                $regions[] = $wBox;
            }
        }

        // Filter tiny regions
        $out = array();
        foreach ($regions as $r) {
            if (count($r['words']) >= $this->minRegionWords) {
                $out[] = $r;
            }
        }

        // If everything filtered out, keep original
        if (count($out) === 0) $out = $regions;

        return $out;
    }

    protected function boxesNear(array $a, array $b, $pad)
    {
        $pad = (int)$pad;
        // Expand a by pad and check overlap with b
        $aL = (int)$a['minLeft'] - $pad;
        $aR = (int)$a['maxRight'] + $pad;
        $aT = (int)$a['minTop'] - $pad;
        $aB = (int)$a['maxBottom'] + $pad;

        $bL = (int)$b['minLeft'];
        $bR = (int)$b['maxRight'];
        $bT = (int)$b['minTop'];
        $bB = (int)$b['maxBottom'];

        $xOverlap = ($bL <= $aR) && ($bR >= $aL);
        $yOverlap = ($bT <= $aB) && ($bB >= $aT);

        return ($xOverlap && $yOverlap);
    }

    protected function mergeBoxes(array $a, array $b)
    {
        $a['minLeft'] = min((int)$a['minLeft'], (int)$b['minLeft']);
        $a['maxRight'] = max((int)$a['maxRight'], (int)$b['maxRight']);
        $a['minTop'] = min((int)$a['minTop'], (int)$b['minTop']);
        $a['maxBottom'] = max((int)$a['maxBottom'], (int)$b['maxBottom']);

        foreach ($b['words'] as $w) $a['words'][] = $w;

        return $a;
    }

    protected function detectTitleBlock(array $regions)
    {
        if (count($regions) < 2) return null;

        // Compute overall bounds
        $minTop = null; $maxBottom = null; $minLeft = null; $maxRight = null;
        foreach ($regions as $r) {
            if ($minTop === null || $r['minTop'] < $minTop) $minTop = $r['minTop'];
            if ($maxBottom === null || $r['maxBottom'] > $maxBottom) $maxBottom = $r['maxBottom'];
            if ($minLeft === null || $r['minLeft'] < $minLeft) $minLeft = $r['minLeft'];
            if ($maxRight === null || $r['maxRight'] > $maxRight) $maxRight = $r['maxRight'];
        }
        if ($minTop === null) return null;

        $height = max(1, $maxBottom - $minTop);
        $width = max(1, $maxRight - $minLeft);

        $best = null;
        $bestScore = 0.0;

        for ($i = 0; $i < count($regions); $i++) {
            $r = $regions[$i];

            $cx = ($r['minLeft'] + $r['maxRight']) / 2.0;
            $cy = ($r['minTop'] + $r['maxBottom']) / 2.0;

            $rightness = ($cx - $minLeft) / $width;  // 0..1
            $bottomness = ($cy - $minTop) / $height; // 0..1

            $area = max(1, ($r['maxRight'] - $r['minLeft']) * ($r['maxBottom'] - $r['minTop']));
            $density = count($r['words']) / $area; // tiny but relative

            $score = (0.45 * $bottomness) + (0.35 * $rightness) + (0.20 * min(1.0, $density * 12000));

            if ($score > $bestScore) {
                $bestScore = $score;
                $best = $i;
            }
        }

        // Require it to be reasonably bottom-right
        if ($best !== null && $bestScore >= 0.62) return $best;

        return null;
    }

    protected function orderRegions(array $regions, $titleIdx)
    {
        $meta = array();
        for ($i = 0; $i < count($regions); $i++) {
            $r = $regions[$i];
            $meta[] = array(
                'idx' => $i,
                'cx' => (int)floor(($r['minLeft'] + $r['maxRight']) / 2),
                'cy' => (int)floor(($r['minTop'] + $r['maxBottom']) / 2),
                'isTitle' => ($titleIdx !== null && $i === $titleIdx),
            );
        }

        usort($meta, function ($a, $b) {
            if ($a['isTitle'] && !$b['isTitle']) return 1;
            if (!$a['isTitle'] && $b['isTitle']) return -1;

            if ($a['cy'] === $b['cy']) {
                if ($a['cx'] === $b['cx']) return 0;
                return ($a['cx'] < $b['cx']) ? -1 : 1;
            }
            return ($a['cy'] < $b['cy']) ? -1 : 1;
        });

        $out = array();
        foreach ($meta as $m) $out[] = $m['idx'];
        return $out;
    }

    protected function linesInRegion(array $lines, array $region)
    {
        $out = array();
        $pad = 10;

        foreach ($lines as $ln) {
            $b = $this->bounds->lineBounds($ln);
            if (!$b) continue;

            $xOverlap = ($b['minLeft'] <= $region['maxRight'] + $pad) && ($b['maxRight'] >= $region['minLeft'] - $pad);
            $yOverlap = ($b['minTop'] <= $region['maxBottom'] + $pad) && ($b['maxBottom'] >= $region['minTop'] - $pad);

            if ($xOverlap && $yOverlap) $out[] = $ln;
        }

        return $out;
    }

    public static function cmpWordTopLeft(TsvRow $a, TsvRow $b)
    {
        if ($a->top === $b->top) {
            if ($a->left === $b->left) return 0;
            return ($a->left < $b->left) ? -1 : 1;
        }
        return ($a->top < $b->top) ? -1 : 1;
    }
}
