<?php

namespace boru\ocr\Layout;

use boru\ocr\Tesseract\Tsv\TsvPage;
use boru\ocr\Layout\Detector\LayoutProfileDetector;
use boru\ocr\Layout\Support\BoundsCalculator;
use boru\ocr\Layout\Support\LineGrouper;
use boru\ocr\Layout\Support\LineRenderer;
use boru\ocr\Layout\Support\LayoutDiagnostics;
use boru\ocr\Layout\Strategy\BandedOrderStrategy;
use boru\ocr\Layout\Strategy\LegacyOrderStrategy;
use boru\ocr\Layout\Strategy\AutoDocumentStrategy;
use boru\ocr\Layout\Strategy\DiagramLayoutStrategy;
use boru\ocr\Layout\Detector\TableLayoutDetector;
use boru\ocr\Layout\Strategy\TableLayoutStrategy;
use boru\ocr\Layout\Support\TableLineRenderer;
use boru\ocr\Layout\Support\TableRegionDetector;


/**
 * Facade: Builds readable baseline page text from TSV.
 *
 * This class keeps the public API small while delegating to:
 * - profile detector (document vs diagram)
 * - ordering strategies
 * - diagram region renderer
 */
class TsvLayoutBuilder
{
    /** @var int */
    protected $minConf = 0;

    /** @var bool */
    protected $dropEmpty = true;

    /** @var string auto|document|diagram */
    protected $layoutProfile = 'auto';

    /** @var bool */
    protected $autoDetectTableProfile = true;

    /** @var bool */
    protected $diagnosticsEnabled = true;

    /** @var LayoutDiagnostics|null */
    protected $lastDiagnostics = null;

    /** @var LayoutDiagnostics[] */
    protected $allDiagnostics = [];

    // helpers
    /** @var BoundsCalculator */
    protected $bounds;

    /** @var LineGrouper */
    protected $grouper;

    /** @var LineRenderer */
    protected $renderer;

    // detector
    /** @var LayoutProfileDetector */
    protected $detector;

    // strategies
    /** @var BandedOrderStrategy */
    protected $banded;

    /** @var LegacyOrderStrategy */
    protected $legacy;

    /** @var AutoDocumentStrategy */
    protected $autoDoc;

    /** @var DiagramLayoutStrategy */
    protected $diagram;

    /** @var TableLayoutDetector */
    protected $tableDetector;

    /** @var TableLayoutStrategy */
    protected $table;

    /** @var TableLineRenderer */
    protected $tableRenderer;

    /** @var TableRegionDetector */
    protected $tableRegionDetector;

    /** @var bool */
    protected $detectTableRegions = true;

    /** @var bool */
    protected $tableRegionMarkers = true;

    /** @var callable|null */
    protected $logger = null;

    /** @var LayoutOptions|null */
    protected $layoutOptions = null;


    /**
     * @param array $options
     *   Common:
     *     - minConf (int) default 0
     *     - dropEmpty (bool) default true
     *     - diagnostics (bool) default true
     *     - layoutProfile (string) default auto (auto|document|diagram|table)
     *
     *   Rendering:
     *     - preserveGapSpacing (bool) default false
     *     - gapExtraSpacePx (int) default 12
     *
     *   Document ordering:
     *     - bandTolerancePx (int) default 14
     *     - headerBandPx (int) default 300
     *     - multiColumn (bool) default true
     *     - detectMixedLayout (bool) default true
     *
     *   Table regions:
     *     - detectTableRegions (bool) default true
     *     - tableRegionMarkers (bool) default false
     *     - tableRegionBucketPx (int) default 18
     *     - tableRegionWindow (int) default 7
     *     - tableRegionMinLines (int) default 5
     *     - tableRegionScoreThreshold (float) default 0.55
     *     - tableRegionMinTokens (int) default 60
     *     - tableRegionMinCols (int) default 3
     *
     *   Diagram:
     *     - diagramScoreThreshold (float) default 0.62
     *     - clusterPadPx (int) default 70
     *     - minRegionWords (int) default 8
     */
    public function __construct(LayoutOptions $options = null)
    {
        $layoutOptions = LayoutOptions::create($options);
        $this->layoutOptions = $layoutOptions;

        if ($this->layoutOptions->logger) $this->logger = $this->layoutOptions->logger;
        $this->minConf = $layoutOptions->minConf;
        $this->dropEmpty = $layoutOptions->dropEmpty;
        $this->diagnosticsEnabled = $layoutOptions->diagnostics;
        $this->layoutProfile = $layoutOptions->layoutProfile;
        $this->autoDetectTableProfile = $layoutOptions->autoDetectTableProfile;
        $this->detectTableRegions = $layoutOptions->detectTableRegions;
        $this->tableRegionMarkers = $layoutOptions->tableRegionMarkers;

        if (!in_array($this->layoutProfile, array('auto','document','diagram','table'), true)) $this->layoutProfile = 'auto';
        
        $this->bounds = new BoundsCalculator();
        $this->grouper = new LineGrouper($layoutOptions);
        $this->renderer = new LineRenderer($layoutOptions);
        $this->tableRegionDetector = new TableRegionDetector($this->bounds, $layoutOptions);

        $this->detector = new LayoutProfileDetector($this->bounds, $layoutOptions);
        $this->tableDetector = new TableLayoutDetector($this->bounds, $layoutOptions);
        $this->tableRenderer = new TableLineRenderer($layoutOptions);
        $this->table = new TableLayoutStrategy($this->bounds,  $layoutOptions);


        $this->banded = new BandedOrderStrategy($this->bounds, $layoutOptions);
        $this->legacy = new LegacyOrderStrategy($this->bounds, $layoutOptions);
        $this->autoDoc = new AutoDocumentStrategy($this->bounds, $this->banded, $this->legacy, $layoutOptions);
        $this->diagram = new DiagramLayoutStrategy($this->bounds, $this->banded, $this->renderer, $layoutOptions);
    }

    /**
     * @return array|null diagnostics array
     */
    public function getLastDiagnostics()
    {
        if (!$this->lastDiagnostics) return null;
        return $this->lastDiagnostics->toArray();
    }

    /**
     * @return array|null diagnostics array
     */
    public function getAllDiagnostics()
    {
        if (empty($this->allDiagnostics)) return null;
        $out = array();
        foreach ($this->allDiagnostics as $d) {
            $out[] = $d->toArray();
        }
        return $out;
    }
    /**
     * Main entrypoint.
     *
     * @param TsvPage $page
     * @return string
     */
    public function buildPageText(TsvPage $page)
    {
        $this->lastDiagnostics = null;

        $diag = new LayoutDiagnostics();
        $diag->page = (int)$page->pageNumber;

        // 1) Select rows (prefer words), apply confidence filtering
        $rows = $this->selectRows($page);

        // 2) Group into lines
        $lines = $this->grouper->groupIntoLines($rows);

        // 3) Decide profile
        $profile = $this->layoutProfile;
        $decision = null;

        if ($profile === 'auto') {
            $decision = $this->detector->detect($page, $rows, $lines);
            $profile = $decision->profile;
            $diag->profileDecision = $decision->toArray();
            $this->log("Profile decision: " . json_encode($diag->profileDecision));
        } else {
            $diag->profileDecision = array('profile' => $profile, 'confidence' => 1.0, 'features' => array(), 'notes' => array('Forced by layoutProfile option.'));
        }

        $diag->profile = $profile;

        // If auto-profile chose 'document', allow a table-dominant override
        if ($this->layoutProfile === 'auto' && $this->autoDetectTableProfile && $profile === 'document') {
            $tableDecision = $this->tableDetector->detect($page, $lines);
            $diag->metrics['tablePageDecision'] = $tableDecision->toArray();
            if ($tableDecision->isTable) {
                $profile = 'table';
                $diag->profile = $profile;
                $this->log("Table decision: " . json_encode($diag->metrics['tablePageDecision']) . " (overriding document)");
            }
        }

        // 4) Build output
        if ($profile === 'diagram') {
            $text = $this->diagram->buildText($rows, $lines, $diag);
        } elseif ($profile === 'table') {
            // full-page table mode
            $ordered = $this->table->order($lines, $diag);
            $out = array();
            foreach ($ordered as $ln) {
                $t = trim($this->tableRenderer->render($ln));
                if ($t !== '') $out[] = $t;
            }
            $text = implode("\n", $out);
        } else {
            // document (optionally lift table-like regions)
            $ordered = $this->autoDoc->order($lines, $diag);

            if ($this->detectTableRegions) {
                $regions = $this->tableRegionDetector->detect($ordered);
                if (!isset($diag->metrics['tableCandidates'])) $diag->metrics['tableCandidates'] = array();

                // Build quick lookup: lineIndex => regionId
                $lineToRegion = array();
                $rid = 0;
                foreach ($regions as $rg) {
                    $this->log("Table candidate: " . json_encode($rg));
                    for ($i = (int)$rg['start']; $i <= (int)$rg['end']; $i++) {
                        $lineToRegion[$i] = $rid;
                    }
                    $cand = $rg;
                    $cand['sourceKind'] = 'region'; // 'kind' is reserved for table kind (grid|key_value)
                    $cand['profile'] = 'document';
                    $cand['region'] = null;
                    $cand['page'] = (int)$page->pageNumber;
                    $diag->metrics['tableCandidates'][] = $cand;
                    $rid++;
                }

                $out = array();
                $openRid = null;

                for ($i = 0; $i < count($ordered); $i++) {
                    $ln = $ordered[$i];
                    $inTable = array_key_exists($i, $lineToRegion);
                    $thisRid = $inTable ? $lineToRegion[$i] : null;

                    if ($this->tableRegionMarkers) {
                        if ($inTable && $openRid === null) {
                            $out[] = '--- table ---';
                            $openRid = $thisRid;
                        } elseif (!$inTable && $openRid !== null) {
                            $out[] = '--- /table ---';
                            $openRid = null;
                        } elseif ($inTable && $openRid !== null && $openRid !== $thisRid) {
                            $out[] = '--- /table ---';
                            $out[] = '--- table ---';
                            $openRid = $thisRid;
                        }
                    }

                    $t = $inTable
                        ? trim($this->tableRenderer->render($ln))
                        : trim($this->renderer->renderLine($ln));

                    if ($t !== '') $out[] = $t;
                }
                if ($this->tableRegionMarkers && $openRid !== null) {
                    $out[] = '--- /table ---';
                }
                $text = implode("\n", $out);
            } else {
                $out = array();
                foreach ($ordered as $ln) {
                    $t = trim($this->renderer->renderLine($ln));
                    if ($t !== '') $out[] = $t;
                }
                $text = implode("\n", $out);
            }
        }

        if ($this->diagnosticsEnabled) {
            $this->lastDiagnostics = $diag;
            $this->allDiagnostics[] = $diag;
        }

        return $text;
    }

    /**
     * Prefer word rows when available; filter empty and confidence.
     *
     * @param TsvPage $page
     * @return array
     */
    protected function selectRows(TsvPage $page)
    {
        $words = $page->words();
        $rows = ($words && count($words) > 0) ? $words : $page->rows;

        $out = array();
        foreach ($rows as $r) {
            if ($this->dropEmpty && $r->text === '') continue;

            if ((int)$r->level === 5) {
                if ($r->conf < $this->minConf) continue;
            }

            $out[] = $r;
        }

        return $out;
    }

    protected function log($msg)
    {
        if ($this->logger) {
            call_user_func($this->logger, $msg);
        }
    }
}
