<?php

namespace boru\ocr\Layout\Strategy;

use boru\ocr\Layout\Support\BoundsCalculator;
use boru\ocr\Layout\Support\LayoutDiagnostics;
use boru\ocr\Layout\Support\LineRenderer;
use boru\ocr\Layout\Support\TableLineRenderer;
use boru\ocr\Layout\Support\TableRegionDetector;
use boru\ocr\Layout\LayoutOptions;
use boru\ocr\Tesseract\Tsv\TsvRow;

/**
 * Diagram layout:
 * - cluster words into spatial regions
 * - detect title block (dense cluster near bottom/right)
 * - render as region blocks, preserving local ordering
 */
class DiagramLayoutStrategy_old
{
    /** @var BoundsCalculator */
    protected $bounds;

    /** @var BandedOrderStrategy */
    protected $banded;

    /** @var LineRenderer */
    protected $renderer;

    /** @var int px */
    protected $clusterPadPx = 70;

    /** @var int */
    protected $minRegionWords = 8;

    /** @var TableRegionDetector */
    protected $tableRegionDetector;

    /** @var TableLineRenderer */
    protected $tableRenderer;

    /** @var bool */
    protected $detectDiagramTableRegions = true;

    /** @var bool */
    protected $diagramTableRegionMarkers = false;

    /** @var bool */
    protected $diagramRegionMarkers = false;

    /** @var LayoutOptions */
    protected $layoutOptions = null;

    /** @var OCRLogger|null */
    protected $logger = null;
    use \boru\ocr\Traits\OcrLogTrait;

    public function __construct(BoundsCalculator $bounds, BandedOrderStrategy $banded, LineRenderer $renderer, LayoutOptions $options = null)
    {
        $this->layoutOptions = LayoutOptions::create($options);
        $this->logger = $this->layoutOptions->logger;
        $this->bounds = $bounds;
        $this->banded = $banded;
        $this->renderer = $renderer;

        $this->clusterPadPx = $this->layoutOptions->clusterPadPx;
        $this->minRegionWords = $this->layoutOptions->minRegionWords;

        $this->detectDiagramTableRegions = $this->layoutOptions->detectTableRegions;
        $this->diagramTableRegionMarkers = $this->layoutOptions->tableRegionMarkers;
        $this->diagramRegionMarkers = $this->layoutOptions->diagramRegionMarkers;

        $this->tableRegionDetector = new TableRegionDetector($this->bounds, $this->layoutOptions);
        $this->tableRenderer = new TableLineRenderer($this->layoutOptions);
    }

    /**
     * Render diagram baseline text.
     *
     * @param TsvRow[] $wordRows
     * @param array<int, TsvRow[]> $lines
     * @param LayoutDiagnostics|null $diag
     * @return string
     */
    public function buildText(array $wordRows, array $lines, LayoutDiagnostics $diag = null)
    {
        $regions = $this->clusterWords($wordRows);

        // fallback if clustering didn't produce meaningful regions
        if (count($regions) <= 1) {
            $ordered = $this->banded->order($lines, null);
            $out = array();
            foreach ($ordered as $ln) {
                $t = trim($this->renderer->renderLine($ln));
                if ($t !== '') $out[] = $t;
            }
            if ($diag) {
                $diag->strategy = 'diagram:fallback-banded';
                $diag->metrics['regionCount'] = count($regions);
            }
            $this->logInfo("Diagram layout fallback to banded ordering, region count: " . count($regions));
            return implode("\n", $out);
        }

        // NEW: suppress equivalent regions (prevents duplicate output)
        $out = $this->suppressEquivalentRegionsUsingLines($regions, $lines);

        // detect title block region
        $titleIdx = $this->detectTitleBlock($regions);

        // NEW: if title block exists, drop any other region that is mostly the same box / text
        if ($titleIdx !== null) {
            $regions = $this->suppressRegionsOverlappingTitleBlock($regions, $titleIdx, $lines);
            $titleIdx = $this->detectTitleBlock($regions); // re-index
        }

        // order regions by centroid top->bottom, left->right, but keep title block last
        $orderedRegions = $this->orderRegions($regions, $titleIdx);

        $out = array();
        $i = 1;

        foreach ($orderedRegions as $ri) {
            $r = $regions[$ri];
            $label = ($ri === $titleIdx) ? 'TITLE BLOCK' : ('REGION ' . $i);
            if ($ri !== $titleIdx) $i++;
            if($this->diagramRegionMarkers) {
                $out[] = '[' . $label . ' @ x=' . $r['minLeft'] . ',y=' . $r['minTop'] . ',w=' . ($r['maxRight'] - $r['minLeft']) . ',h=' . ($r['maxBottom'] - $r['minTop']) . ']';
            }

            // Build lines for region: filter original lines whose bounds intersect region
            $regionLines = $this->linesInRegion($lines, $r);

            $regionLines = $this->banded->order($regionLines, null);

            // Optionally lift table-like sub-regions inside this diagram region (BOM/title blocks/etc.)
            if ($this->detectDiagramTableRegions) {
                // Detect table-like regions inside this diagram region.
                // IMPORTANT: Detect using spatial ordering (top/left), then map onto rendered order (banded) by bbox.
                $regionLinesForDetect = $regionLines;

                // sort by bbox top, then left
                $meta = array();
                for ($di = 0; $di < count($regionLinesForDetect); $di++) {
                    $bb = $this->bounds->lineBounds($regionLinesForDetect[$di]);
                    if ($bb === null) continue;
                    $meta[] = array(
                        'idx'  => $di,
                        'top'  => (int)$bb['top'],
                        'left' => (int)$bb['left'],
                    );
                }
                usort($meta, function($a, $b) {
                    if ($a['top'] === $b['top']) {
                        if ($a['left'] === $b['left']) return 0;
                        return ($a['left'] < $b['left']) ? -1 : 1;
                    }
                    return ($a['top'] < $b['top']) ? -1 : 1;
                });

                $orderedForDetect = array();
                foreach ($meta as $m) {
                    $orderedForDetect[] = $regionLinesForDetect[$m['idx']];
                }

                // detect on spatial order
                $regions2 = $this->tableRegionDetector->detect($orderedForDetect, 'diagram');
                // Filter noisy candidates (diagram pages generate lots of false positives)
                $minScore = 0.65; // tweak 0.60–0.75 as needed
                $regions2Filtered = array();
                foreach ($regions2 as $rg) {
                    $score = isset($rg['score']) ? (float)$rg['score'] : 0.0;

                    // prefer grid tables a bit more than kv, but still require score
                    if ($score < $minScore) continue;

                    // require bbox
                    if (!isset($rg['bbox']) || $rg['bbox'] === null) continue;

                    // require minimum height (avoid tiny spans)
                    $bb = $rg['bbox'];
                    if ((int)$bb['h'] < 60) continue;

                    $regions2Filtered[] = $rg;
                }
                $regions2 = $regions2Filtered;


                // diagnostics (keep kind/profile/region label as you had)
                if ($diag) {
                    if (!isset($diag->metrics['tableCandidates'])) $diag->metrics['tableCandidates'] = array();
                    foreach ($regions2 as $rg) {
                        $cand = $rg;
                        $cand['sourceKind'] = 'region'; // avoid colliding with 'kind' = grid/key_value
                        $cand['profile'] = 'diagram';
                        $cand['region'] = $label;
                        $diag->metrics['tableCandidates'][] = $cand;
                    }
                }

                // Build bbox list for mapping onto output order (banded regionLines)
                $regionBboxes = array();
                $r2id = 0;
                foreach ($regions2 as $rg) {
                    if (isset($rg['bbox']) && is_array($rg['bbox'])) {
                        $bb = $rg['bbox'];
                        $regionBboxes[$r2id] = array(
                            'x' => (int)$bb['x'],
                            'y' => (int)$bb['y'],
                            'w' => (int)$bb['w'],
                            'h' => (int)$bb['h'],
                        );
                    }
                    $r2id++;
                }

                // pick region id by bbox containment of line center
                $pickRegion2 = function($lineBb) use (&$regionBboxes) {
                    if ($lineBb === null || empty($regionBboxes)) return null;

                    $cx = (int)round($lineBb['left'] + $lineBb['width'] / 2.0);
                    $cy = (int)round($lineBb['top']  + $lineBb['height'] / 2.0);

                    $bestRid  = null;
                    $bestArea = null;

                    foreach ($regionBboxes as $rid2 => $rb) {
                        $pad = 4;
                        $x1 = (int)$rb['x'] - $pad;
                        $y1 = (int)$rb['y'] - $pad;
                        $x2 = (int)$rb['x'] + (int)$rb['w'] + $pad;
                        $y2 = (int)$rb['y'] + (int)$rb['h'] + $pad;

                        if ($cx >= $x1 && $cx <= $x2 && $cy >= $y1 && $cy <= $y2) {
                            $area = (int)$rb['w'] * (int)$rb['h'];
                            if ($bestArea === null || $area < $bestArea) {
                                $bestArea = $area;
                                $bestRid  = $rid2;
                            }
                        }
                    }
                    return $bestRid;
                };

                // Render in banded order, but decide inTable by bbox
                $open = null;
                $pendingOpen = false;
                $pendingRid = null;

                for ($k = 0; $k < count($regionLines); $k++) {
                    $ln = $regionLines[$k];
                    $lnBb = $this->bounds->lineBounds($ln);

                    $thisRid = $pickRegion2($lnBb);
                    $inTable = ($thisRid !== null);

                    // handle marker state, but don't output yet
                    if ($this->diagramTableRegionMarkers) {
                        if ($inTable && $open === null && !$pendingOpen) {
                            $pendingOpen = true;
                            $pendingRid = $thisRid;
                        } elseif (!$inTable) {
                            // close any open table (only if it was actually opened)
                            if ($open !== null) {
                                $out[] = '--- /table ---';
                                $open = null;
                            }
                            // also cancel pending open if we never emitted anything
                            $pendingOpen = false;
                            $pendingRid = null;
                        } elseif ($inTable && $open !== null && $open !== $thisRid) {
                            $out[] = '--- /table ---';
                            $open = null;
                            $pendingOpen = true;
                            $pendingRid = $thisRid;
                        }
                    }

                    $t = $inTable
                        ? trim($this->tableRenderer->render($ln))
                        : trim($this->renderer->renderLine($ln));

                    if ($t === '') continue;

                    // If this is the first actual content inside a pending table, open it now
                    if ($this->diagramTableRegionMarkers && $pendingOpen) {
                        $out[] = '--- table ---';
                        $open = $pendingRid;
                        $pendingOpen = false;
                        $pendingRid = null;
                    }

                    $out[] = $t;
                }

                // close at end if actually open
                if ($this->diagramTableRegionMarkers && $open !== null) {
                    $out[] = '--- /table ---';
                }

            } else {
                foreach ($regionLines as $ln) {
                    $t = trim($this->renderer->renderLine($ln));
                    if ($t === '') continue;
                    $out[] = $t;
                }
            }

            $out[] = '';
        }

        if ($diag) {
            $diag->strategy = 'diagram:regions';
            $diag->metrics['regionCount'] = count($regions);
            $diag->metrics['titleBlockDetected'] = ($titleIdx !== null);
        }

        return rtrim(implode("\n", $out));
    }

    /**
     * Cluster word rows into region boxes using greedy bbox merging.
     *
     * @param TsvRow[] $rows
     * @return array<int,array{minLeft:int,maxRight:int,minTop:int,maxBottom:int,words:array}>
     */
    protected function clusterWords(array $rows)
    {
        $words = array();
        foreach ($rows as $r) {
            if ((int)$r->level !== 5) continue;
            if ($r->text === '') continue;
            if ($r->conf < 0) continue;
            $words[] = $r;
        }
        if (count($words) === 0) return array();

        // Sort by top then left for stable greedy merging
        usort($words, array(__CLASS__, 'cmpWordTopLeft'));

        $regions = array();

        foreach ($words as $w) {
            $wBox = array(
                'minLeft' => (int)$w->left,
                'maxRight' => (int)$w->right(),
                'minTop' => (int)$w->top,
                'maxBottom' => (int)$w->top + (int)$w->height,
                'words' => array($w),
            );

            $mergedInto = null;

            // Try to merge into an existing region if close
            for ($i = 0; $i < count($regions); $i++) {
                if ($this->boxesNear($regions[$i], $wBox, $this->clusterPadPx)) {
                    $regions[$i] = $this->mergeBoxes($regions[$i], $wBox);
                    $mergedInto = $i;
                    break;
                }
            }

            if ($mergedInto === null) {
                $regions[] = $wBox;
            }
        }

        // Filter tiny regions
        $out = array();
        foreach ($regions as $r) {
            if (count($r['words']) >= $this->minRegionWords) {
                $out[] = $r;
            }
        }

        // If everything filtered out, keep original
        if (count($out) === 0) $out = $regions;

        // NEW: suppress nested/overlapping regions (prevents duplicate output)
        $out = $this->suppressContainedRegions($out);

        return $out;
    }

    protected function boxesNear(array $a, array $b, $pad)
    {
        $pad = (int)$pad;
        // Expand a by pad and check overlap with b
        $aL = (int)$a['minLeft'] - $pad;
        $aR = (int)$a['maxRight'] + $pad;
        $aT = (int)$a['minTop'] - $pad;
        $aB = (int)$a['maxBottom'] + $pad;

        $bL = (int)$b['minLeft'];
        $bR = (int)$b['maxRight'];
        $bT = (int)$b['minTop'];
        $bB = (int)$b['maxBottom'];

        $xOverlap = ($bL <= $aR) && ($bR >= $aL);
        $yOverlap = ($bT <= $aB) && ($bB >= $aT);

        return ($xOverlap && $yOverlap);
    }

    protected function mergeBoxes(array $a, array $b)
    {
        $a['minLeft'] = min((int)$a['minLeft'], (int)$b['minLeft']);
        $a['maxRight'] = max((int)$a['maxRight'], (int)$b['maxRight']);
        $a['minTop'] = min((int)$a['minTop'], (int)$b['minTop']);
        $a['maxBottom'] = max((int)$a['maxBottom'], (int)$b['maxBottom']);

        foreach ($b['words'] as $w) $a['words'][] = $w;

        return $a;
    }

    protected function detectTitleBlock(array $regions)
    {
        if (count($regions) < 2) return null;

        // Compute overall bounds
        $minTop = null; $maxBottom = null; $minLeft = null; $maxRight = null;
        foreach ($regions as $r) {
            if ($minTop === null || $r['minTop'] < $minTop) $minTop = $r['minTop'];
            if ($maxBottom === null || $r['maxBottom'] > $maxBottom) $maxBottom = $r['maxBottom'];
            if ($minLeft === null || $r['minLeft'] < $minLeft) $minLeft = $r['minLeft'];
            if ($maxRight === null || $r['maxRight'] > $maxRight) $maxRight = $r['maxRight'];
        }
        if ($minTop === null) return null;

        $height = max(1, $maxBottom - $minTop);
        $width = max(1, $maxRight - $minLeft);

        $best = null;
        $bestScore = 0.0;

        for ($i = 0; $i < count($regions); $i++) {
            $r = $regions[$i];

            $cx = ($r['minLeft'] + $r['maxRight']) / 2.0;
            $cy = ($r['minTop'] + $r['maxBottom']) / 2.0;

            $rightness = ($cx - $minLeft) / $width;  // 0..1
            $bottomness = ($cy - $minTop) / $height; // 0..1

            $area = max(1, ($r['maxRight'] - $r['minLeft']) * ($r['maxBottom'] - $r['minTop']));
            $density = count($r['words']) / $area; // tiny but relative

            $score = (0.45 * $bottomness) + (0.35 * $rightness) + (0.20 * min(1.0, $density * 12000));

            if ($score > $bestScore) {
                $bestScore = $score;
                $best = $i;
            }
        }

        // Require it to be reasonably bottom-right
        if ($best !== null && $bestScore >= 0.62) return $best;

        return null;
    }

    protected function orderRegions(array $regions, $titleIdx)
    {
        $meta = array();
        for ($i = 0; $i < count($regions); $i++) {
            $r = $regions[$i];
            $meta[] = array(
                'idx' => $i,
                'cx' => (int)floor(($r['minLeft'] + $r['maxRight']) / 2),
                'cy' => (int)floor(($r['minTop'] + $r['maxBottom']) / 2),
                'isTitle' => ($titleIdx !== null && $i === $titleIdx),
            );
        }

        usort($meta, function ($a, $b) {
            if ($a['isTitle'] && !$b['isTitle']) return 1;
            if (!$a['isTitle'] && $b['isTitle']) return -1;

            if ($a['cy'] === $b['cy']) {
                if ($a['cx'] === $b['cx']) return 0;
                return ($a['cx'] < $b['cx']) ? -1 : 1;
            }
            return ($a['cy'] < $b['cy']) ? -1 : 1;
        });

        $out = array();
        foreach ($meta as $m) $out[] = $m['idx'];
        return $out;
    }

    protected function linesInRegion(array $lines, array $region)
    {
        $out = array();
        $pad = 10;

        foreach ($lines as $ln) {
            $b = $this->bounds->lineBounds($ln);
            if (!$b) continue;

            $xOverlap = ($b['minLeft'] <= $region['maxRight'] + $pad) && ($b['maxRight'] >= $region['minLeft'] - $pad);
            $yOverlap = ($b['minTop'] <= $region['maxBottom'] + $pad) && ($b['maxBottom'] >= $region['minTop'] - $pad);

            if ($xOverlap && $yOverlap) $out[] = $ln;
        }

        return $out;
    }

    /**
     * Suppress regions that are mostly contained within another region.
     * This prevents duplicate text output when clustering produces nested boxes.
     *
     * @param array<int,array> $regions
     * @return array<int,array>
     */
    protected function suppressContainedRegions(array $regions)
    {
        if (count($regions) <= 1) return $regions;

        // Sort largest->smallest so we keep big containers first
        usort($regions, array(__CLASS__, 'cmpRegionAreaDesc'));

        $kept = array();

        foreach ($regions as $r) {
            $drop = false;

            foreach ($kept as $k) {
                // If r is mostly inside k, drop r
                $ratio = $this->containmentRatio($r, $k); // intersection / area(r)
                if ($ratio >= 0.88) { // tune 0.80–0.92
                    $drop = true;
                    break;
                }
            }

            if (!$drop) $kept[] = $r;
        }

        return $kept;
    }

    /**
     * Suppress regions that are "siblings" but contain essentially the same text.
     * This handles diagrams where the same legend/table header gets clustered
     * into multiple side-by-side regions.
     *
     * @param array $regions region boxes from clusterWords()
     * @param array<int, array> $lines array<int, TsvRow[]>
     * @return array filtered regions
     */
    protected function suppressEquivalentRegionsUsingLines(array $regions, array $lines)
    {
        if (count($regions) <= 1) return $regions;

        // Sort by area desc so we keep the most "complete" region first
        usort($regions, array(__CLASS__, 'cmpRegionAreaDesc'));

        $kept = array();
        $keptSigs = array(); // cache signatures for kept regions

        foreach ($regions as $r) {
            // Build the lines inside this region once
            $regionLines = $this->linesInRegion($lines, $r);
            $sig = $this->regionTextSignatureFromLines($regionLines);

            // empty sig => keep (it might be mostly graphics and later becomes useful)
            if ($sig === '') {
                $kept[] = $r;
                $keptSigs[] = $sig;
                continue;
            }

            $drop = false;

            // Compare against already kept regions
            for ($i = 0; $i < count($kept); $i++) {
                $k = $kept[$i];

                // Must overlap vertically a lot to be considered sibling duplicates
                if ($this->verticalOverlapRatio($r, $k) < 0.85) continue;

                $kSig = $keptSigs[$i];
                if ($kSig === '') continue;

                // Similarity (PHP built-in)
                similar_text($sig, $kSig, $pct);

                if ($pct >= 90.0) {
                    $drop = true;
                    break;
                }
            }

            if (!$drop) {
                $kept[] = $r;
                $keptSigs[] = $sig;
            }
        }

        return $kept;
    }

    protected function regionTextSignatureFromLines(array $regionLines)
    {
        $buf = array();

        foreach ($regionLines as $ln) {
            $t = trim($this->renderer->renderLine($ln));
            if ($t === '') continue;

            // normalize hard: collapse whitespace, lowercase, strip obvious noise
            $t = preg_replace('/\s+/u', ' ', $t);
            $t = mb_strtolower($t);

            // Optional: drop single-char junk lines that dominate diagrams
            if (mb_strlen($t) <= 1) continue;

            $buf[] = $t;
        }

        if (empty($buf)) return '';

        // Limit size so similar_text doesn't get expensive on huge regions
        $sig = implode("\n", $buf);
        if (strlen($sig) > 8000) {
            $sig = substr($sig, 0, 8000);
        }

        return $sig;
    }

    protected function verticalOverlapRatio(array $a, array $b)
    {
        $iy = max(
            0,
            min((int)$a['maxBottom'], (int)$b['maxBottom']) - max((int)$a['minTop'], (int)$b['minTop'])
        );

        $ha = max(1, (int)$a['maxBottom'] - (int)$a['minTop']);
        $hb = max(1, (int)$b['maxBottom'] - (int)$b['minTop']);

        return $iy / min($ha, $hb);
    }

    protected static function cmpRegionAreaDesc($a, $b)
    {
        $aa = max(1, ((int)$a['maxRight'] - (int)$a['minLeft']) * ((int)$a['maxBottom'] - (int)$a['minTop']));
        $bb = max(1, ((int)$b['maxRight'] - (int)$b['minLeft']) * ((int)$b['maxBottom'] - (int)$b['minTop']));
        if ($aa === $bb) return 0;
        return ($aa > $bb) ? -1 : 1;
    }

    /**
     * How much of $inner lies inside $outer, as intersection_area / area(inner).
     */
    protected function containmentRatio(array $inner, array $outer)
    {
        $ix = max(0, min((int)$inner['maxRight'], (int)$outer['maxRight']) - max((int)$inner['minLeft'], (int)$outer['minLeft']));
        $iy = max(0, min((int)$inner['maxBottom'], (int)$outer['maxBottom']) - max((int)$inner['minTop'], (int)$outer['minTop']));
        $inter = $ix * $iy;

        $areaInner = max(1, ((int)$inner['maxRight'] - (int)$inner['minLeft']) * ((int)$inner['maxBottom'] - (int)$inner['minTop']));
        return $inter / $areaInner;
    }

    protected function regionTextSignature(array $r, array $lines)
    {
        $buf = array();

        foreach ($lines as $ln) {
            $b = $this->bounds->lineBounds($ln);
            if (!$b) continue;

            if (
                $b['minLeft'] >= $r['minLeft'] &&
                $b['maxRight'] <= $r['maxRight'] &&
                $b['minTop'] >= $r['minTop'] &&
                $b['maxBottom'] <= $r['maxBottom']
            ) {
                $t = trim($this->renderer->renderLine($ln));
                if ($t !== '') {
                    // normalize aggressively
                    $buf[] = preg_replace('/\s+/u', ' ', $t);
                }
            }
        }

        return mb_strtolower(implode("\n", $buf));
    }

    /**
 * Drop regions that overlap the TITLE BLOCK heavily (to prevent duplicate footer/title content).
 * Keeps the title block, removes near-duplicates.
 */
protected function suppressRegionsOverlappingTitleBlock(array $regions, $titleIdx, array $lines)
{
    if ($titleIdx === null) return $regions;
    if (!isset($regions[$titleIdx])) return $regions;

    $tb = $regions[$titleIdx];

    // Build a strong signature for the title block
    $tbLines = $this->linesInRegion($lines, $tb);
    $tbSig = $this->regionTextSignatureFromLines($tbLines);

    $out = array();
    foreach ($regions as $i => $r) {
        if ($i === $titleIdx) {
            $out[] = $r;
            continue;
        }

        // If region is largely inside title block or overlaps massively, consider it duplicate
        $contain = $this->containmentRatio($r, $tb); // intersection/area(r)
        $vOverlap = $this->verticalOverlapRatio($r, $tb);

        if ($contain >= 0.85 || $vOverlap >= 0.90) {
            // If text is also extremely similar, drop it
            $rLines = $this->linesInRegion($lines, $r);
            $rSig = $this->regionTextSignatureFromLines($rLines);

            if ($tbSig !== '' && $rSig !== '') {
                similar_text($tbSig, $rSig, $pct);
                if ($pct >= 92.0) {
                    continue; // DROP duplicate
                }
            } else {
                // no signature available; still drop if bbox overlap is huge
                if ($contain >= 0.90) continue;
            }
        }

        $out[] = $r;
    }

    // We changed array indexes; titleIdx is now stale.
    // That's okay: orderRegions() uses $titleIdx only for "keep title last".
    // Recompute titleIdx by finding the same bbox in the new array:
    // easiest approach: store title block separately in buildText() and pass it in,
    // but if you want minimal change, just detect again after suppression.
    return $out;
}



    public static function cmpWordTopLeft(TsvRow $a, TsvRow $b)
    {
        if ($a->top === $b->top) {
            if ($a->left === $b->left) return 0;
            return ($a->left < $b->left) ? -1 : 1;
        }
        return ($a->top < $b->top) ? -1 : 1;
    }
}
