<?php

namespace boru\ocr\Agent;

use boru\ocr\OcrEngineResult;
use boru\ocr\OCRLogger;
use boru\ocr\Traits\OcrLogTrait;

class AiOcrOrchestrator
{
    /** @var bool */
    protected $usePlanner = true;

    /** @var bool */
    protected $useTableInterpreter = true;

    protected $plannerOptions;
    protected $tableInterpreterOptions;
    protected $finalAgentOptions;

    /** @var OCRLogger|null */
    protected $logger;
    use OcrLogTrait;

    /**
     * @param array $options
     *  - usePlanner (bool) default true
     *  - logger (OCRLogger|null) default null
     * @param OCRLogger|null $logger
     */
    public function __construct(AiOptions $options = null, OCRLogger $logger = null)
    {
        if ($options === null) {
            $options = new AiOptions();
        }
        $this->usePlanner = $options->usePlanner;
        $this->useTableInterpreter = $options->userTableInterpreter;
        $this->plannerOptions = $options->plannerOptions;
        $this->tableInterpreterOptions = $options->tableInterpreterOptions;
        $this->finalAgentOptions = $options->finalAgentOptions;
        if ($logger !== null) {
            $this->logger = $logger;
        } elseif ($options->logger instanceof OCRLogger) {
            $this->logger = $options->logger;
        } else {
            $this->logger = new OCRLogger();
        }
    }

    /**
     * @param string $pdfPath
     * @param OcrEngineResult $bundle
     * @param array $plannerOptions
     * @param array $finalAgentOptions
     * @return string final framed output
     */
    public function run($pdfPath, OcrEngineResult $bundle)
    {
        if(!$this->plannerOptions) {
            $this->plannerOptions = new AgentOptions();
            $this->plannerOptions->logger = $this->logger;
        }
        if(!$this->tableInterpreterOptions) {
            $this->tableInterpreterOptions = new AgentOptions();
            $this->tableInterpreterOptions->logger = $this->logger; 
        }
        if(!$this->finalAgentOptions) {
            $this->finalAgentOptions = new AgentOptions();
            $this->finalAgentOptions->logger = $this->logger; 
        }
        // Build TSV tools using the correct FunctionBase/ToolDefinition system.
        $bridge = new EvidenceToolBridge($bundle->evidence());
        $tools = $bridge->buildTools();
        $plan = null;
        if ($this->usePlanner) {
            $planner = new PlannerAgent($pdfPath, $this->plannerOptions, $this->logger);
            $plan = $planner->run($bundle);
        }

        $tableJson = null;
        if ($this->useTableInterpreter && !empty($bundle->tableCandidates())) {
            $tableAgent = new TableInterpreterAgent($pdfPath, $tools, $this->tableInterpreterOptions, $this->logger);
            $tableJson = $tableAgent->run($bundle, $plan);
        }

        $final = new FinalAgent($pdfPath, $tools, $this->finalAgentOptions, $this->logger);
        return $final->run($bundle, $plan, $tableJson);
    }
}
