<?php

namespace boru\ocr\Agent\Functions;

use boru\boruai\Models\Functions\FunctionBase;
use boru\boruai\Models\ToolDefinition;

use boru\ocr\Evidence\TsvEvidenceIndex;
use boru\ocr\Evidence\TsvTextMatch;
use boru\ocr\OCRLogger;

/**
 * TSV evidence tools exposed to the AI runtime via BoruAI's FunctionBase/ToolDefinition system.
 *
 * Pattern matches Browse.php: build ToolDefinition, register closure calling static::callback(...) 
 */
class TsvEvidenceTools extends FunctionBase
{
    /** @var TsvEvidenceIndex|null */
    protected static $evidence = null;

    protected static $logger = null;

    /**
     * Backward compatibility alias for functionDefinitions()
     * @return ToolDefinition[]
     */
    public static function functionDefintions()
    {
        return static::functionDefintions();
    }

    /**
     * Must be called before functionDefinitions() is used.
     *
     * @param TsvEvidenceIndex $evidence
     * @return void
     */
    public static function setEvidence(TsvEvidenceIndex $evidence)
    {
        static::$evidence = $evidence;
    }

    /**
     * Set a logger for debugging.
     *
     * @param OCRLogger $logger
     * @return void
     */
    public static function setLogger(OCRLogger $logger)
    {
        static::$logger = $logger;
    }

    /**
     * @return ToolDefinition[]
     */
    public static function functionDefinitions()
    {
        $return = array();

        // Safety: if evidence isn't set, return no tools.
        if (!static::$evidence) {
            return $return;
        }

        // -------------------------
        // tsv_find_text
        // -------------------------
        $toolDefinition = new ToolDefinition();
        $toolDefinition->name("tsv_find_text");
        $toolDefinition->description(
            "Find occurrences of a text anchor in TSV-derived lines. Returns matches with page, bbox, and lineKey."
        );
        $toolDefinition->addString("needle", "Text to search for (case-insensitive match in line text).");
        $toolDefinition->addInteger("page", "Optional 1-based page number to constrain the search.", array(), false);
        $toolDefinition->addInteger("minConf", "Minimum word confidence to include in search (0-100).", array(), false);
        $toolDefinition->addInteger("maxMatches", "Maximum number of matches to return.", array(), false);
        $toolDefinition->register(function ($tool) {
            return static::callback("tsv_find_text", $tool);
        });
        $return[] = $toolDefinition;

        // -------------------------
        // tsv_get_box_text
        // -------------------------
        $toolDefinition = new ToolDefinition();
        $toolDefinition->name("tsv_get_box_text");
        $toolDefinition->description(
            "Return TSV-derived text inside a bounding box on a page (x/y are pixel coords relative to the rendered page)."
        );
        $toolDefinition->addInteger("page", "1-based page number.");
        $toolDefinition->addInteger("x1", "Left X coordinate (px).");
        $toolDefinition->addInteger("y1", "Top Y coordinate (px).");
        $toolDefinition->addInteger("x2", "Right X coordinate (px).");
        $toolDefinition->addInteger("y2", "Bottom Y coordinate (px).");
        $toolDefinition->addInteger("minConf", "Minimum word confidence to include (0-100).", array(), false);
        $toolDefinition->register(function ($tool) {
            return static::callback("tsv_get_box_text", $tool);
        });
        $return[] = $toolDefinition;

        // -------------------------
        // tsv_get_near_match_text
        // -------------------------
        $toolDefinition = new ToolDefinition();
        $toolDefinition->name("tsv_get_near_match_text");
        $toolDefinition->description(
            "Return TSV-derived text near an anchor bbox (adds padding around the bbox before extracting)."
        );
        $toolDefinition->addInteger("page", "1-based page number.");
        $toolDefinition->addInteger("left", "Anchor bbox left (px).");
        $toolDefinition->addInteger("top", "Anchor bbox top (px).");
        $toolDefinition->addInteger("width", "Anchor bbox width (px).");
        $toolDefinition->addInteger("height", "Anchor bbox height (px).");
        $toolDefinition->addString("lineKey", "Optional line key in format block:par:line (for debugging).", array(), false);
        $toolDefinition->addInteger("padX", "Horizontal padding (px) around bbox.", array(), false);
        $toolDefinition->addInteger("padY", "Vertical padding (px) around bbox.", array(), false);
        $toolDefinition->addInteger("minConf", "Minimum word confidence to include (0-100).", array(), false);
        $toolDefinition->register(function ($tool) {
            return static::callback("tsv_get_near_match_text", $tool);
        });
        $return[] = $toolDefinition;

        // -------------------------
        // tsv_get_line_text
        // -------------------------
        $toolDefinition = new ToolDefinition();
        $toolDefinition->name("tsv_get_line_text");
        $toolDefinition->description(
            "Return TSV-derived text for a specific TSV line key (block:par:line) on a page."
        );
        $toolDefinition->addInteger("page", "1-based page number.");
        $toolDefinition->addString("lineKey", "Line key in format block:par:line.");
        $toolDefinition->addInteger("minConf", "Minimum word confidence to include (0-100).", array(), false);
        $toolDefinition->register(function ($tool) {
            return static::callback("tsv_get_line_text", $tool);
        });
        $return[] = $toolDefinition;

        return $return;
    }

    // ============================================================
    // Tool callbacks (static methods invoked via FunctionBase::callback)
    // Pattern: $args = $tool->get(); like Browse::genericHttpRequest :contentReference[oaicite:4]{index=4}
    // ============================================================

    public static function tsv_find_text($tool)
    {
        static::log("tsv_find_text called with args: " . json_encode($tool->get()));
        $args = $tool->get();
        $needle = isset($args['needle']) ? (string)$args['needle'] : '';
        if ($needle === '') {
            return array('error' => 'Missing needle');
        }

        $opts = array();
        if (isset($args['page']) && (int)$args['page'] > 0) {
            $opts['page'] = (int)$args['page'];
        }
        if (isset($args['minConf'])) {
            $opts['minConf'] = (int)$args['minConf'];
        }
        if (isset($args['maxMatches'])) {
            $opts['maxMatches'] = (int)$args['maxMatches'];
        }

        $matches = static::$evidence->findText($needle, $opts);

        $out = array();
        foreach ($matches as $m) {
            $out[] = array(
                'page' => $m->page,
                'lineKey' => $m->lineKey,
                'left' => $m->left,
                'top' => $m->top,
                'width' => $m->width,
                'height' => $m->height,
                'text' => $m->text,
            );
        }

        return array(
            'needle' => $needle,
            'count' => count($out),
            'matches' => $out,
        );
    }

    public static function tsv_get_box_text($tool)
    {
        static::log("tsv_get_box_text called with args: " . json_encode($tool->get()));
        $args = $tool->get();
        $page = isset($args['page']) ? (int)$args['page'] : 0;
        $x1 = isset($args['x1']) ? (int)$args['x1'] : 0;
        $y1 = isset($args['y1']) ? (int)$args['y1'] : 0;
        $x2 = isset($args['x2']) ? (int)$args['x2'] : 0;
        $y2 = isset($args['y2']) ? (int)$args['y2'] : 0;

        if ($page <= 0) {
            return array('error' => 'Invalid page');
        }

        $minConf = isset($args['minConf']) ? (int)$args['minConf'] : 0;

        $text = static::$evidence->getBoxText($page, $x1, $y1, $x2, $y2, array(
            'minConf' => $minConf,
            'useWordsOnly' => true,
            'sort' => 'reading',
        ));

        return array(
            'page' => $page,
            'box' => array('x1' => $x1, 'y1' => $y1, 'x2' => $x2, 'y2' => $y2),
            'minConf' => $minConf,
            'text' => $text,
        );
    }

    public static function tsv_get_near_match_text($tool)
    {
        static::log("tsv_get_near_match_text called with args: " . json_encode($tool->get()));
        $args = $tool->get();

        $page = isset($args['page']) ? (int)$args['page'] : 0;
        $left = isset($args['left']) ? (int)$args['left'] : 0;
        $top = isset($args['top']) ? (int)$args['top'] : 0;
        $width = isset($args['width']) ? (int)$args['width'] : 0;
        $height = isset($args['height']) ? (int)$args['height'] : 0;

        if ($page <= 0) {
            return array('error' => 'Invalid page');
        }

        $padX = isset($args['padX']) ? (int)$args['padX'] : 120;
        $padY = isset($args['padY']) ? (int)$args['padY'] : 80;
        $minConf = isset($args['minConf']) ? (int)$args['minConf'] : 0;

        $match = new TsvTextMatch(array(
            'page' => $page,
            'left' => $left,
            'top' => $top,
            'width' => $width,
            'height' => $height,
            'lineKey' => isset($args['lineKey']) ? (string)$args['lineKey'] : '',
            'text' => '',
        ));

        $text = static::$evidence->getNearMatchText($match, array(
            'padX' => $padX,
            'padY' => $padY,
            'minConf' => $minConf,
        ));

        return array(
            'page' => $page,
            'padX' => $padX,
            'padY' => $padY,
            'minConf' => $minConf,
            'text' => $text,
        );
    }

    public static function tsv_get_line_text($tool)
    {
        static::log("tsv_get_line_text called with args: " . json_encode($tool->get()));
        $args = $tool->get();

        $page = isset($args['page']) ? (int)$args['page'] : 0;
        $lineKey = isset($args['lineKey']) ? (string)$args['lineKey'] : '';

        if ($page <= 0 || $lineKey === '') {
            return array('error' => 'Missing page or lineKey');
        }

        $minConf = isset($args['minConf']) ? (int)$args['minConf'] : 0;

        $rows = static::$evidence->getLineRows($page, $lineKey, array(
            'minConf' => $minConf,
            'useWordsOnly' => true,
        ));

        $parts = array();
        foreach ($rows as $r) {
            if ($r->text !== '') $parts[] = $r->text;
        }

        return array(
            'page' => $page,
            'lineKey' => $lineKey,
            'minConf' => $minConf,
            'wordCount' => count($parts),
            'text' => trim(implode(' ', $parts)),
        );
    }

    protected static function log($msg)
    {
        if (static::$logger) {
            static::$logger->output("TsvEvidenceTools",OCRLogger::$LEVEL_INFO,$msg);
        }
    }
}
