<?php
namespace boru\ocr\Cli;

use boru\cli2\CLIContext;
use boru\cli2\Contracts\CommandDescriptionInterface;
use boru\cli2\Contracts\CommandParamsInterface;
use boru\cli2\Contracts\CommandInterface;
use boru\cli2\Params\Flag;
use boru\cli2\Params\Positional;
use boru\cli2\Params\Option;
use boru\ocr\OcrEngine;

class TesseractCommand implements CommandInterface, CommandParamsInterface, CommandDescriptionInterface {
    public function name(): string {
        return "tesseract";
    }
    public function description(): string {
        return "Run Tesseract OCR on an image file and output the text.";
    }

    public function commandParams(): array {
        return [
            Positional::create("file|Path to the image file to OCR"),
            Flag::create("z|debug|Enable debug output"),
            Option::create("i|image-save-dir|Directory to save intermediate images"),
            Option::create("d|outputdir|Path to save the OCR output text file(s)"),
        ];
    }

    public function handle(CLIContext $ctx): int {
        $file = $ctx->get("file");
        $debug = $ctx->get("debug", false);
        $outputDir = $ctx->get("outputdir", null);
        $imageSaveDir = $ctx->get("image-save-dir", null);

        if (!file_exists($file)) {
            fwrite(STDERR, "File not found: {$file}\n");
            return 1;
        }
        $basename = basename($file);
        $outputDir = CliCommon::ensureOutputDir($outputDir);
        $tmpDir = CliCommon::ensureTempDir('boru_ocr');

        $engine = OcrEngine::forFile($file);
        $engine->withImageDir($tmpDir);

        if($debug) {
            $logger = CliCommon::getLogger(true, true, true, true);
        } else {
            $logger = CliCommon::getLogger(false,false,false,true);
        }
        $engine->withLogger($logger);
        
        $tesseractOptions = [
            "lang" => "eng",
            "config" => [
                "preserve_interword_spaces" => "1",
            ],
        ];
        if ($imageSaveDir !== null) {
            $tesseractOptions['saveImages'] = $imageSaveDir;
        }
        $engine->withTesseractOptions($tesseractOptions);
        $ocrEngineResult = $engine->run();
        $ctx->output("OCR completed. Saving output...");
        $ocrEngineResult->saveDebugOutput($outputDir . '/' . $basename . '_debug');
        $ctx->output("Saved to " . $outputDir . '/' . $basename . '_debug');

        return 0;
    }

}