<?php
namespace boru\ocr\Cli;

use boru\cli2\CLIContext;
use boru\cli2\Contracts\CommandDescriptionInterface;
use boru\cli2\Contracts\CommandParamsInterface;
use boru\cli2\Contracts\CommandInterface;
use boru\cli2\Params\Flag;
use boru\cli2\Params\Positional;
use boru\cli2\Params\Option;
use boru\ocr\Layout\LayoutOptions;
use boru\ocr\OcrEngine;

class DebugCommand implements CommandInterface, CommandParamsInterface, CommandDescriptionInterface {
    public function name() {
        return "debug";
    }
    public function description() {
        return "Run OCR engine with debug options and save intermediate outputs.";
    }

    public function commandParams() {
        return [
            Positional::create("file|Path to the image file to OCR"),
            Flag::create("z|debug|Enable debug output"),
            Option::create("i|image-save-dir|Directory to save intermediate images"),
            Option::create("d|outputdir|Path to save the OCR output text file(s)"),
            Option::create("I|ai|Enable AI OCR orchestrator and save intermediate outputs"),
            Option::create("P|no-planner|Disable planner agent in AI OCR orchestrator"),
        ];
    }

    public function handle(CLIContext $ctx) {
        $file = $ctx->get("file");
        $debug = $ctx->get("debug", false);
        $outputDir = $ctx->get("outputdir", null);
        $imageSaveDir = $ctx->get("image-save-dir", null);
        $useAi = $ctx->get("ai", false);
        $usePlanner = !$ctx->get("no-planner", false);
        if (!file_exists($file)) {
            fwrite(STDERR, "File not found: {$file}\n");
            return 1;
        }
        $basename = basename($file);
        $outputDir = CliCommon::ensureOutputDir($outputDir);
        $tmpDir = CliCommon::ensureTempDir('boru_ocr');

        $engine = OcrEngine::forFile($file);
        $engine->withImageDir($tmpDir);

        if($debug) {
            $logger = CliCommon::getLogger(true, true, true, true);
        } else {
            $logger = CliCommon::getLogger(false,false,false,true);
        }
        $engine->withLogger($logger);
        
        if($usePlanner) {
            if($ctx->get('boruai_is_init', false) === false) {
                fwrite(STDERR, "BoruAI is not initialized. Please provide a valid config file with --config option.\n");
                return 1;
            }
            $engine->withPlanner();
        }
        if($useAi) {
            if($ctx->get('boruai_is_init', false) === false) {
                fwrite(STDERR, "BoruAI is not initialized. Please provide a valid config file with --config option.\n");
                return 1;
            }
            $engine->withAI();
        }

        $tesseractOptions = [
            "lang" => "eng",
            "config" => [
                "preserve_interword_spaces" => "1",
            ],
        ];
        if ($imageSaveDir !== null) {
            $tesseractOptions['saveImages'] = $imageSaveDir;
        }
        $engine->withTesseractOptions($tesseractOptions);
        $ocrEngineResult = $engine->run();
        $ctx->output("OCR completed. Saving output...");
        $ocrEngineResult->saveDebugOutput($outputDir . '/' . $basename . '_debug');
        $ctx->output("Saved to " . $outputDir . '/' . $basename . '_debug');

        return 0;
    }

}