<?php
namespace boru\ocr;

class OCRLogger {
    public static $LEVEL_LOG = 'log';
    public static $LEVEL_ERROR = 'error';
    public static $LEVEL_DEBUG = 'debug';
    public static $LEVEL_INFO = 'info';

    protected $namePad = 25;
    protected $endNewLine = true;
    protected $eol = PHP_EOL;

    protected $enabledNames = [];
    protected $enabledLevels = [];
    protected $disabledNames = [];
    protected $disabledLevels = [];

    protected $debugCallback = null;
    protected $errorCallback = null;
    protected $infoCallback = null;
    protected $logCallback = null;

    public function __construct($options = []) {
        $defaults = $this->defaults();
        $options = array_merge($defaults, $options);
        $this->setOptions($options);
    }

    public function setOptions($options = []) {
        foreach ($options as $key => $value) {
            if (property_exists($this, $key)) {
                if($key == 'enabledNames') {
                    if(is_array($value)) {
                        foreach($value as $name) {
                            $this->enableName($name);
                        }
                    }
                } elseif($key == 'enabledLevels') {
                    if(is_array($value)) {
                        foreach($value as $level) {
                            $this->enableLevel($level);
                        }
                    }
                } elseif($key == 'disabledNames') {
                    if(is_array($value)) {
                        foreach($value as $name) {
                            $this->disableName($name);
                        }
                    }
                } elseif($key == 'disabledLevels') {
                    if(is_array($value)) {
                        foreach($value as $level) {
                            $this->disableLevel($level);
                        }
                    }
                } else {
                    $this->$key = $value;
                }
            }
        }
        return $this;
    }

    public function defaults() {
        return [
            'namePad' => 15,
            'endNewLine' => true,
            'eol' => PHP_EOL,
            'enabledNames' => [],
            'enabledLevels' => [],
            'disabledNames' => [],
            'disabledLevels' => [
                self::$LEVEL_LOG,
                self::$LEVEL_ERROR,
                self::$LEVEL_DEBUG,
                self::$LEVEL_INFO,
            ],
            'debugCallback' => null,
            'errorCallback' => null,
            'infoCallback' => null,
            'logCallback' => null,
        ];
    }

    public function output($name,$level="info",$message="") {
        if(!$this->shouldLog($name, $level)) {
            return;
        }
        $message = $this->format($name, $message);
        if($level == self::$LEVEL_LOG) {
            $this->log($message);
        } elseif($level == self::$LEVEL_ERROR) {
            $this->error($message);
        } elseif($level == self::$LEVEL_DEBUG) {
            $this->debug($message);
        } elseif($level == self::$LEVEL_INFO) {
            $this->info($message);
        }
    }
    public function log($message) {
        if($this->logCallback) {
            call_user_func($this->logCallback, $message);
        } else {
            echo $this->padLevel(self::$LEVEL_LOG) . $message;
        }
    }
    public function error($message) {
        if($this->errorCallback) {
            call_user_func($this->errorCallback, $message);
        } else {
            echo $this->padLevel(self::$LEVEL_ERROR) . $message;
        }
    }
    public function debug($message) {
        if($this->debugCallback) {
            call_user_func($this->debugCallback, $message);
        } else {
            echo $this->padLevel(self::$LEVEL_DEBUG) . $message;
        }
    }
    public function info($message) {
        if($this->infoCallback) {
            call_user_func($this->infoCallback, $message);
        } else {
            echo $this->padLevel(self::$LEVEL_INFO) . $message;
        }
    }



    public function enableName($name) {
        if(in_array($name, $this->disabledNames)) {
            $index = array_search($name, $this->disabledNames);
            unset($this->disabledNames[$index]);
        }
        if(!in_array($name, $this->enabledNames)) {
            $this->enabledNames[] = $name;
        }
        return $this;
    }
    public function enableLevel($level) {
        if(in_array($level, $this->disabledLevels)) {
            $index = array_search($level, $this->disabledLevels);
            unset($this->disabledLevels[$index]);
        }
        if(!in_array($level, $this->enabledLevels)) {
            $this->enabledLevels[] = $level;
        }
        return $this;
    }
    public function disableName($name) {
        if(in_array($name, $this->enabledNames)) {
            $index = array_search($name, $this->enabledNames);
            unset($this->enabledNames[$index]);
        }
        if(!in_array($name, $this->disabledNames)) {
            $this->disabledNames[] = $name;
        }
        return $this;
    }
    public function disableLevel($level) {
        if(in_array($level, $this->enabledLevels)) {
            $index = array_search($level, $this->enabledLevels);
            unset($this->enabledLevels[$index]);
        }
        if(!in_array($level, $this->disabledLevels)) {
            $this->disabledLevels[] = $level;
        }
        return $this;
    }
    public function setDebugCallback($callback) {
        $this->debugCallback = $callback;
        return $this;
    }
    public function setErrorCallback($callback) {
        $this->errorCallback = $callback;
        return $this;
    }
    public function setInfoCallback($callback) {
        $this->infoCallback = $callback;
        return $this;
    }
    public function setLogCallback($callback) {
        $this->logCallback = $callback;
        return $this;
    }
    public function setNamePad($pad) {
        $this->namePad = $pad;
        return $this;
    }
    public function setEol($eol) {
        $this->eol = $eol;
        return $this;
    }
    public function setEndNewLine($endNewLine) {
        $this->endNewLine = $endNewLine ? true : false;
        return $this;
    }


    private function shouldLog($name, $level) {
        return $this->shouldLogName($name) && $this->shouldLogLevel($level);
    }
    private function shouldLogName($name) {
        if (!empty($this->enabledNames) && !in_array($name, $this->enabledNames)) {
            return false;
        }
        if (in_array($name, $this->disabledNames)) {
            return false;
        }
        return true;
    }
    private function shouldLogLevel($level) {
        if (!empty($this->enabledLevels) && !in_array($level, $this->enabledLevels)) {
            return false;
        }
        if (in_array($level, $this->disabledLevels)) {
            return false;
        }
        return true;
    }

    private function padLevel($level) {
        return str_pad("[" . strtoupper($level) . "]", 10, ' ', STR_PAD_RIGHT);
    }
    private function format($name, $message) {
        $name = str_pad($name, $this->namePad, ' ', STR_PAD_RIGHT);
        if($this->endNewLine && substr($message, -strlen($this->eol)) !== $this->eol) {
            $message = $message . $this->eol;
        }
        return $name . ' ' . $message;
    }


    //modern-ish with* style setters:
    public function withDebugCallback($callback) {
        return $this->setDebugCallback($callback);
    }
    public function withErrorCallback($callback) {
        return $this->setErrorCallback($callback);
    }
    public function withInfoCallback($callback) {
        return $this->setInfoCallback($callback);
    }
    public function withLogCallback($callback) {
        return $this->setLogCallback($callback);
    }
    public function withNamePad($pad) {
        return $this->setNamePad($pad);
    }
    public function withEol($eol) {
        return $this->setEol($eol);
    }
    public function withEndNewLine($endNewLine) {
        return $this->setEndNewLine($endNewLine);
    }
}