<?php

namespace boru\ocr\Layout\Diagram;

class DiagramRegionOrderer
{
    /**
     * Detect likely title block region (dense cluster near bottom/right).
     *
     * @param array<int, array> $regions
     * @return int|null
     */
    public function detectTitleBlock(array $regions)
    {
        if (count($regions) < 2) return null;

        $minTop = null; $maxBottom = null; $minLeft = null; $maxRight = null;
        foreach ($regions as $r) {
            if ($minTop === null || $r['minTop'] < $minTop) $minTop = $r['minTop'];
            if ($maxBottom === null || $r['maxBottom'] > $maxBottom) $maxBottom = $r['maxBottom'];
            if ($minLeft === null || $r['minLeft'] < $minLeft) $minLeft = $r['minLeft'];
            if ($maxRight === null || $r['maxRight'] > $maxRight) $maxRight = $r['maxRight'];
        }
        if ($minTop === null) return null;

        $height = max(1, $maxBottom - $minTop);
        $width = max(1, $maxRight - $minLeft);

        $best = null;
        $bestScore = 0.0;

        for ($i = 0; $i < count($regions); $i++) {
            $r = $regions[$i];

            $cx = ($r['minLeft'] + $r['maxRight']) / 2.0;
            $cy = ($r['minTop'] + $r['maxBottom']) / 2.0;

            $rightness = ($cx - $minLeft) / $width;
            $bottomness = ($cy - $minTop) / $height;

            $area = max(1, ($r['maxRight'] - $r['minLeft']) * ($r['maxBottom'] - $r['minTop']));
            $density = count($r['words']) / $area;

            $score = (0.45 * $bottomness) + (0.35 * $rightness) + (0.20 * min(1.0, $density * 12000));

            if ($score > $bestScore) {
                $bestScore = $score;
                $best = $i;
            }
        }

        if ($best !== null && $bestScore >= 0.62) return $best;

        return null;
    }

    /**
     * Order regions by centroid top->bottom, left->right, with title block last.
     *
     * @param array<int, array> $regions
     * @param int|null $titleIdx
     * @return array<int,int> list of region indices
     */
    public function orderRegions(array $regions, $titleIdx)
    {
        $meta = array();
        for ($i = 0; $i < count($regions); $i++) {
            $r = $regions[$i];
            $meta[] = array(
                'idx' => $i,
                'cx' => (int)floor(($r['minLeft'] + $r['maxRight']) / 2),
                'cy' => (int)floor(($r['minTop'] + $r['maxBottom']) / 2),
                'isTitle' => ($titleIdx !== null && $i === $titleIdx),
            );
        }

        usort($meta, function ($a, $b) {
            if ($a['isTitle'] && !$b['isTitle']) return 1;
            if (!$a['isTitle'] && $b['isTitle']) return -1;

            if ($a['cy'] === $b['cy']) {
                if ($a['cx'] === $b['cx']) return 0;
                return ($a['cx'] < $b['cx']) ? -1 : 1;
            }
            return ($a['cy'] < $b['cy']) ? -1 : 1;
        });

        $out = array();
        foreach ($meta as $m) $out[] = $m['idx'];
        return $out;
    }
}
