<?php

namespace boru\ocr\Layout\Strategy;

use boru\ocr\Layout\Support\BoundsCalculator;
use boru\ocr\Layout\Support\LayoutDiagnostics;
use boru\ocr\Layout\Support\LineRenderer;
use boru\ocr\Layout\Support\TableLineRenderer;
use boru\ocr\Layout\Support\TableRegionDetector;
use boru\ocr\Layout\LayoutOptions;
use boru\ocr\Layout\Support\RegionLineSelector;
use boru\ocr\Layout\Support\RegionSignature;
use boru\ocr\Layout\Diagram\DiagramRegionClusterer;
use boru\ocr\Layout\Diagram\DiagramRegionOrderer;
use boru\ocr\Layout\Diagram\DiagramRegionDeduper;
use boru\ocr\Tesseract\Tsv\TsvRow;

/**
 * Diagram layout:
 * - cluster words into spatial regions
 * - detect title block (dense cluster near bottom/right)
 * - render as region blocks, preserving local ordering
 */
class DiagramLayoutStrategy
{
    /** @var BoundsCalculator */
    protected $bounds;

    /** @var BandedOrderStrategy */
    protected $banded;

    /** @var LineRenderer */
    protected $renderer;

    /** @var int px */
    protected $clusterPadPx = 70;

    /** @var int */
    protected $minRegionWords = 8;

    /** @var bool */
    protected $detectDiagramTableRegions = false;

    /** @var TableRegionDetector */
    protected $tableRegionDetector;

    /** @var TableLineRenderer */
    protected $tableRenderer;

    /** @var bool */
    protected $diagramTableRegionMarkers = false;

    /** @var bool */
    protected $diagramRegionMarkers = false;

    /** @var LayoutOptions */
    protected $layoutOptions = null;

    /** @var OCRLogger|null */
    protected $logger = null;
    use \boru\ocr\Traits\OcrLogTrait;

    /** @var DiagramRegionClusterer */
    protected $clusterer;

    /** @var DiagramRegionOrderer */
    protected $orderer;

    /** @var DiagramRegionDeduper */
    protected $deduper;

    /** @var RegionLineSelector */
    protected $lineSelector;

    /** @var RegionSignature */
    protected $signature;

    public function __construct(BoundsCalculator $bounds, BandedOrderStrategy $banded, LineRenderer $renderer, LayoutOptions $options = null)
    {
        $this->layoutOptions = LayoutOptions::create($options);
        $this->logger = $this->layoutOptions->logger;

        $this->bounds = $bounds;
        $this->banded = $banded;
        $this->renderer = $renderer;

        $this->clusterPadPx = $this->layoutOptions->clusterPadPx;
        $this->minRegionWords = $this->layoutOptions->minRegionWords;

        $this->detectDiagramTableRegions = $this->layoutOptions->detectTableRegions;
        $this->diagramTableRegionMarkers = $this->layoutOptions->tableRegionMarkers;
        $this->diagramRegionMarkers = $this->layoutOptions->diagramRegionMarkers;

        $this->tableRegionDetector = new TableRegionDetector($this->bounds, $this->layoutOptions);
        $this->tableRenderer = new TableLineRenderer($this->layoutOptions);

        // New split components
        $this->lineSelector = new RegionLineSelector($this->bounds, 10);
        $this->signature = new RegionSignature($this->renderer, 8000);

        $this->clusterer = new DiagramRegionClusterer($this->layoutOptions);
        $this->orderer = new DiagramRegionOrderer();
        $this->deduper = new DiagramRegionDeduper($this->lineSelector, $this->signature);
    }

    /**
     * Render diagram baseline text.
     *
     * @param TsvRow[] $wordRows
     * @param array<int, TsvRow[]> $lines
     * @param LayoutDiagnostics|null $diag
     * @return string
     */
    public function buildText(array $wordRows, array $lines, LayoutDiagnostics $diag = null)
    {
        $regions = $this->clusterer->clusterWords($wordRows);

        // fallback if clustering didn't produce meaningful regions
        if (count($regions) <= 1) {
            $ordered = $this->banded->order($lines, null);
            $out = array();
            foreach ($ordered as $ln) {
                $t = trim($this->renderer->renderLine($ln));
                if ($t !== '') $out[] = $t;
            }
            if ($diag) {
                $diag->strategy = 'diagram:fallback-banded';
                $diag->metrics['regionCount'] = count($regions);
            }
            $this->logInfo("Diagram layout fallback to banded ordering, region count: " . count($regions));
            return implode("\n", $out);
        }

        // Suppress nested regions (containment) + equivalent sibling regions
        $regions = $this->deduper->suppressContainedRegions($regions);
        $regions = $this->deduper->suppressEquivalentRegionsUsingLines($regions, $lines);

        // detect title block region
        $titleIdx = $this->orderer->detectTitleBlock($regions);

        // Drop any other region that is mostly the same as title block
        if ($titleIdx !== null) {
            $regions = $this->deduper->suppressRegionsOverlappingTitleBlock($regions, $titleIdx, $lines);
            $titleIdx = $this->orderer->detectTitleBlock($regions); // re-index
        }

        // order regions by centroid top->bottom, left->right, but keep title block last
        $orderedRegions = $this->orderer->orderRegions($regions, $titleIdx);

        $out = array();
        $i = 1;

        foreach ($orderedRegions as $ri) {
            $r = $regions[$ri];
            $label = ($ri === $titleIdx) ? 'TITLE BLOCK' : ('REGION ' . $i);
            if ($ri !== $titleIdx) $i++;

            if ($this->diagramRegionMarkers) {
                $out[] = '[' . $label . ' @ x=' . $r['minLeft'] . ',y=' . $r['minTop'] . ',w=' . ($r['maxRight'] - $r['minLeft']) . ',h=' . ($r['maxBottom'] - $r['minTop']) . ']';
            }

            // Build lines for region: filter original lines whose bounds intersect region
            $regionLines = $this->lineSelector->linesInRegion($lines, $r);
            $regionLines = $this->banded->order($regionLines, null);

            // Optionally lift table-like sub-regions inside this diagram region (BOM/title blocks/etc.)
            if ($this->detectDiagramTableRegions) {
                $regions2 = $this->tableRegionDetector->detect($regionLines, 'diagram');

                if ($diag) {
                    if (!isset($diag->metrics['tableCandidates'])) $diag->metrics['tableCandidates'] = array();
                    foreach ($regions2 as $rg) {
                        $cand = $rg;
                        $cand['kind'] = 'region';
                        $cand['profile'] = 'diagram';
                        $cand['region'] = $label;
                        $diag->metrics['tableCandidates'][] = $cand;
                    }
                }

                // Build lookup line->region2
                $lineToR2 = array();
                $r2id = 0;
                foreach ($regions2 as $rg) {
                    for ($k = (int)$rg['start']; $k <= (int)$rg['end']; $k++) {
                        $lineToR2[$k] = $r2id;
                    }
                    $r2id++;
                }

                $open = null;
                for ($k = 0; $k < count($regionLines); $k++) {
                    $ln = $regionLines[$k];
                    $inTable = array_key_exists($k, $lineToR2);
                    $thisRid = $inTable ? $lineToR2[$k] : null;

                    if ($this->diagramTableRegionMarkers) {
                        if ($inTable && $open === null) {
                            $out[] = '--- table ---';
                            $open = $thisRid;
                        } elseif (!$inTable && $open !== null) {
                            $out[] = '--- /table ---';
                            $open = null;
                        } elseif ($inTable && $open !== null && $open !== $thisRid) {
                            $out[] = '--- /table ---';
                            $out[] = '--- table ---';
                            $open = $thisRid;
                        }
                    }

                    $t = $inTable
                        ? trim($this->tableRenderer->render($ln))
                        : trim($this->renderer->renderLine($ln));

                    if ($t !== '') $out[] = $t;
                }
                if ($this->diagramTableRegionMarkers && $open !== null) {
                    $out[] = '--- /table ---';
                }
            } else {
                foreach ($regionLines as $ln) {
                    $t = trim($this->renderer->renderLine($ln));
                    if ($t === '') continue;
                    $out[] = $t;
                }
            }

            $out[] = '';
        }

        if ($diag) {
            $diag->strategy = 'diagram:regions';
            $diag->metrics['regionCount'] = count($regions);
            $diag->metrics['titleBlockDetected'] = ($titleIdx !== null);
        }

        return rtrim(implode("\n", $out));
    }
}
