<?php
namespace boru\ocr\Cli;

use boru\cli2\CLIContext;
use boru\cli2\Contracts\CommandDescriptionInterface;
use boru\cli2\Contracts\CommandParamsInterface;
use boru\cli2\Contracts\CommandInterface;
use boru\cli2\Params\Option;
use boru\cli2\Params\Flag;

class DockerBuildCommand implements CommandInterface, CommandParamsInterface, CommandDescriptionInterface
{
    public function name()
    {
        return 'docker-build';
    }

    public function description()
    {
        return 'Build the boru/ocr-tools Docker image from Docker/Dockerfile.';
    }

    public function commandParams()
    {
        return array(
            // Optional tag override: default is boru/ocr-tools:latest
            Option::create('t|tag|Docker image tag (default: boru/ocr-tools:latest)'),
            // Optional build-arg for TESSERACT_VERSION
            Option::create('v|tesseract-version|Tesseract version/branch for build-arg TESSERACT_VERSION (default: main)'),
            // Show the docker build command without running it
            Flag::create('n|dry-run|Only print the docker build command, do not execute it'),
        );
    }

    public function handle(CLIContext $ctx)
    {
        $tag = $ctx->get('tag');
        if (!$tag) {
            $tag = 'boru/ocr-tools:latest';
        }

        $tessVersion = $ctx->get('tesseract-version');
        if (!$tessVersion) {
            $tessVersion = 'main';
        }

        // Resolve repo root as the directory containing this file's ../../..
        $repoRoot = realpath(__DIR__ . '/../..');
        if ($repoRoot === false) {
            $ctx->output("Could not resolve repository root.");
            return 1;
        }

        $dockerfileDir = $repoRoot . '/Docker';
        if (!is_dir($dockerfileDir)) {
            $ctx->output("Docker directory not found at: " . $dockerfileDir);
            return 1;
        }

        $dockerfilePath = $dockerfileDir . '/Dockerfile';
        if (!file_exists($dockerfilePath)) {
            $ctx->output("Dockerfile not found at: " . $dockerfilePath);
            return 1;
        }

        // Build command: docker build -f Docker/Dockerfile -t boru/ocr-tools:latest --build-arg TESSERACT_VERSION=...
        $cmdParts = array(
            'docker',
            'build',
            '-t',
            escapeshellarg($tag),
            $dockerfileDir
        );

        $cmd = implode(' ', $cmdParts);

        $ctx->output("Docker build command:");
        $ctx->output("  " . $cmd);

        if ($ctx->get('dry-run', false)) {
            $ctx->output("Dry run enabled; not executing docker build.");
            return 0;
        }

        $ctx->output("Running docker build, this may take a while...");
        $output = array();
        $exitCode = 0;
        exec($cmd . ' 2>&1', $output, $exitCode);

        foreach ($output as $line) {
            $ctx->output($line);
        }

        if ($exitCode !== 0) {
            $ctx->output("docker build failed with exit code " . $exitCode);
            return $exitCode;
        }

        $ctx->output("docker build completed successfully. Image tagged as: " . $tag);
        return 0;
    }
}
