<?php
namespace boru\ocr\Page;

use boru\ocr\OCRLogger;
use boru\ocr\Traits\OcrLogTrait;

class DockerVipsPdfPageImageProvider implements PageImageProviderInterface
{
    use OcrLogTrait;

    private $pdfPath;
    private $imageDir;

    /** DPI for mutool rendering */
    private $tileDpi = 600;

    /** If 0, never tile. Otherwise tile when width*height > threshold */
    private $tilePixelThreshold = 0;

    private $tileCols = 2;
    private $tileRows = 2;
    private $tileOverlap = 20;

    private $mutoolBandHeight = 0;

    /** NEW: VIPS auto-tiling options (same as VipsPdfPageImageProvider) */
    private $vipsAutoTile = false;
    private $vipsTileSize = 1024;

    /** @var OCRLogger|null */
    protected $logger;

    /**
     * Docker-related options
     */
    private $dockerBinary = 'docker run';
    private $dockerImage = 'boru/ocr-tools:latest';
    private $dockerOptions = '-it --rm';
    private $dockerMountTarget = '/work';

    /**
     * Cached pages.
     *
     * @var array<int,string|array<int,array{path:string,offset_x:int,offset_y:int}>>|null
     */
    private $images = null;

    /**
     * @param string $pdfPath
     * @param string $imageDir
     * @param array<string,mixed> $options
     */
    public function __construct($pdfPath, $imageDir, array $options = array())
    {
        $this->pdfPath  = $pdfPath;
        $this->imageDir = rtrim($imageDir, '/');

        if (!is_dir($this->imageDir)) {
            @mkdir($this->imageDir, 0777, true);
        }

        if (isset($options['logger']) && $options['logger'] instanceof OCRLogger) {
            $this->logger = $options['logger'];
        }

        $this->logInfo("Docker VIPS PDF Page Image Provider initialized.");

        // Copy most of VipsPdfPageImageProvider options:
        if (isset($options['tileDpi'])) {
            $this->tileDpi = max(72, (int)$options['tileDpi']);
        }
        if (isset($options['tilePixelThreshold'])) {
            $this->tilePixelThreshold = (int)$options['tilePixelThreshold'];
        }
        if (isset($options['tileCols'])) {
            $this->tileCols = max(1, (int)$options['tileCols']);
        }
        if (isset($options['tileRows'])) {
            $this->tileRows = max(1, (int)$options['tileRows']);
        }
        if (isset($options['tileOverlap'])) {
            $this->tileOverlap = max(0, (int)$options['tileOverlap']);
        }
        if (isset($options['mutoolBandHeight'])) {
            $this->mutoolBandHeight = max(0, (int)$options['mutoolBandHeight']);
        }
        if (isset($options['vipsAutoTile'])) {
            $this->vipsAutoTile = (bool)$options['vipsAutoTile'];
        }
        if (isset($options['vipsTileSize'])) {
            $this->vipsTileSize = max(64, (int)$options['vipsTileSize']);
        }

        // Docker-specific configuration under $options['dockerVips'] or top-level
        $docker = isset($options['dockerVips']) && is_array($options['dockerVips'])
            ? $options['dockerVips']
            : $options;

        if (isset($docker['dockerBinary']) && is_string($docker['dockerBinary']) && $docker['dockerBinary'] !== '') {
            $this->dockerBinary = $docker['dockerBinary'];
        }
        if (isset($docker['dockerImage']) && is_string($docker['dockerImage']) && $docker['dockerImage'] !== '') {
            $this->dockerImage = $docker['dockerImage'];
        }
        if (isset($docker['dockerOptions']) && is_string($docker['dockerOptions'])) {
            $this->dockerOptions = $docker['dockerOptions'];
        }
        if (isset($docker['dockerMountTarget']) && is_string($docker['dockerMountTarget']) && $docker['dockerMountTarget'] !== '') {
            $this->dockerMountTarget = $docker['dockerMountTarget'];
        }
    }

    public static function isAvailable()
    {
        // Only requirement is that docker (or chosen binary) is present
        $docker = @shell_exec('command -v docker 2>/dev/null');
        if(is_string($docker) && trim($docker) !== '') {
            $ws_line = @shell_exec('docker image ls | grep "boru/ocr-tools" 2>/dev/null');
            if(is_string($ws_line) && strpos($ws_line, 'boru/ocr-tools') !== false) {
                return true;
            }
        }
        return false;
    }

    /**
     * @return array<int,string|array<int,array{path:string,offset_x:int,offset_y:int}>>
     */
    public function getPages()
    {
        if ($this->images !== null) {
            $this->logInfo("Using cached Docker VIPS page images.");
            return $this->images;
        }

        $pageCount = $this->getPageCount();
        if ($pageCount < 1) {
            $pageCount = 1;
        }

        $result = array();

        for ($page = 1; $page <= $pageCount; $page++) {
            $this->logInfo("Rendering page {$page} with docker+mutool+vips...");
            $pngPath = $this->renderPageWithMutool($page);

            if (!file_exists($pngPath)) {
                $this->logError("Failed to render page {$page} via docker mutool+vips. -- $pngPath not found.");
                $result[$page - 1] = "[MUPDF+VIPS RENDER FAILED]";
                continue;
            }

            $size = @getimagesize($pngPath);
            if (!$size) {
                $result[$page - 1] = $pngPath;
                continue;
            }

            $width  = (int)$size[0];
            $height = (int)$size[1];
            $pixels = $width * $height;

            if ($this->tilePixelThreshold > 0 && $pixels > $this->tilePixelThreshold) {
                $this->logInfo("Page {$page} exceeds pixel threshold; creating tiles...");
                if ($this->vipsAutoTile) {
                    $tiles = $this->createTilesWithVipsDzsave($pngPath, $page - 1);
                    if (!empty($tiles)) {
                        $result[$page - 1] = $tiles;
                        continue;
                    }
                    $result[$page - 1] = $this->createTilesWithVipsCrop($pngPath, $page - 1, $width, $height);
                } else {
                    $result[$page - 1] = $this->createTilesWithVipsCrop($pngPath, $page - 1, $width, $height);
                }
            } else {
                $this->logInfo("Page {$page} within pixel threshold; using full page image.");
                $this->logDebug("Pixels: {$pixels}, Threshold: {$this->tilePixelThreshold}");
                $result[$page - 1] = $pngPath;
            }
        }

        $this->images = $result;
        return $this->images;
    }

    private function makeDockerBaseCommand() {
        $cmdParts = array();
        $cmdParts[] = escapeshellcmd($this->dockerBinary);
        if ($this->dockerOptions !== '') {
            $cmdParts[] = $this->dockerOptions;
        }
        $cmdParts[] = '-v';
        $cmdParts[] = escapeshellarg(realpath(dirname($this->pdfPath)) . ':' . $this->dockerMountTarget);
        $cmdParts[] = '-v';
        $cmdParts[] = escapeshellarg(realpath($this->imageDir) . ':' . "/images");
        $cmdParts[] = escapeshellarg($this->dockerImage);

        return $cmdParts;
    }
    private function pdfInContainer() {
        return rtrim($this->dockerMountTarget, '/') . '/' . basename($this->pdfPath);
    }
    private function imageInContainer($imagePath) {
        return rtrim($this->dockerMountTarget, '/') . '/' . basename($imagePath);
    }
    private function outputImageInContainer($outputPath) {
        return '/images/' . ltrim($outputPath, '/');
    }

    /**
     * Get page count using dockerized mutool.
     */
    private function getPageCount()
    {
        $realPdf = realpath($this->pdfPath);
        if ($realPdf === false) {
            $realPdf = $this->pdfPath;
        }

        $cmdParts = $this->makeDockerBaseCommand();

        $cmdParts[] = 'mutool';
        $cmdParts[] = 'info';
        $cmdParts[] = escapeshellarg($this->pdfInContainer());

        $cmd = implode(' ', $cmdParts) . ' 2>/dev/null';

        $output = shell_exec($cmd);
        if (!is_string($output) || $output === '') {
            return 1;
        }

        if (preg_match('/Pages:\\s*([0-9]+)/', $output, $m)) {
            return max(1, (int)$m[1]);
        }
        if (preg_match_all('/^Page\\s+([0-9]+)/mi', $output, $mm)) {
            $count = count($mm[1]);
            if ($count > 0) {
                return $count;
            }
        }

        return 1;
    }

    /**
     * Render a single page PNG via dockerized mutool.
     *
     * @param int $page
     * @return string
     */
    private function renderPageWithMutool($page)
    {
        $pattern = $this->imageDir . '/vips_page_%d.png';
        $outPath = $this->imageDir . '/vips_page_' . $page . '.png';
        @unlink($outPath);

        $realPdf = realpath($this->pdfPath);
        if ($realPdf === false) {
            $realPdf = $this->pdfPath;
        }

        $cmdParts = $this->makeDockerBaseCommand();

        $cmdParts[] = 'mutool';
        $cmdParts[] = 'draw';
        $cmdParts[] = '-F';
        $cmdParts[] = 'png';
        $cmdParts[] = '-r';
        $cmdParts[] = (string)$this->tileDpi;

        if ($this->mutoolBandHeight > 0) {
            $cmdParts[] = '-B';
            $cmdParts[] = (string)$this->mutoolBandHeight;
        }

        $cmdParts[] = '-o';
        $cmdParts[] = escapeshellarg($this->outputImageInContainer('vips_page_%d.png'));
        $cmdParts[] = escapeshellarg($this->pdfInContainer());
        $cmdParts[] = (string)$page;

        $cmd = implode(' ', $cmdParts) . ' 2>/dev/null';
        $this->logInfo("Docker mutool command: {$cmd}");
        shell_exec($cmd);

        return $outPath;
    }

    /**
     * EXISTING behavior: manual tiling via vips crop, but run inside Docker.
     *
     * @param string $pngPath
     * @param int    $pageIndex
     * @param int    $width
     * @param int    $height
     * @return array<int,array{path:string,offset_x:int,offset_y:int}>
     */
    private function createTilesWithVipsCrop($pngPath, $pageIndex, $width, $height)
    {
        $tiles = array();

        $cols    = max(1, $this->tileCols);
        $rows    = max(1, $this->tileRows);
        $overlap = $this->tileOverlap;

        $tileWidth  = (int)ceil($width  / $cols);
        $tileHeight = (int)ceil($height / $rows);

        for ($ty = 0; $ty < $rows; $ty++) {
            for ($tx = 0; $tx < $cols; $tx++) {
                $x = $tx * $tileWidth;
                $y = $ty * $tileHeight;

                $x = max(0, $x - $overlap);
                $y = max(0, $y - $overlap);

                $wTile = min($tileWidth + $overlap * 2, $width  - $x);
                $hTile = min($tileHeight + $overlap * 2, $height - $y);

                if ($wTile <= 0 || $hTile <= 0) {
                    continue;
                }

                $tilePath = sprintf(
                    '%s/page_%d_tile_%d_%d.png',
                    $this->imageDir,
                    $pageIndex,
                    $tx,
                    $ty
                );
                @unlink($tilePath);

                // Build docker + vips crop command
                $cmdParts = $this->makeDockerBaseCommand();

                $cmdParts[] = 'vips';
                $cmdParts[] = 'crop';

                // inside-container paths: source and dest both under dockerMountTarget,
                // but since we have the host path, we can just use basename here,
                // because the whole $imageDir is under the same mount.
                $cmdParts[] = escapeshellarg($this->outputImageInContainer($pngPath));
                $cmdParts[] = escapeshellarg($this->outputImageInContainer($tilePath));
                $cmdParts[] = (string)$x;
                $cmdParts[] = (string)$y;
                $cmdParts[] = (string)$wTile;
                $cmdParts[] = (string)$hTile;

                $cmd = implode(' ', $cmdParts) . ' 2>/dev/null';
                $this->logInfo("Docker VIPS crop tile command: {$cmd}");
                shell_exec($cmd);

                if (!file_exists($tilePath)) {
                    continue;
                }

                $tiles[] = array(
                    'path'     => $tilePath,
                    'offset_x' => $x,
                    'offset_y' => $y,
                );
            }
        }

        return $tiles;
    }


    /**
     * VIPS DeepZoom auto-tiling via dzsave, run inside Docker.
     *
     * @param string $pngPath  full-page image
     * @param int    $pageIndex zero-based page index
     * @return array<int,array{path:string,offset_x:int,offset_y:int}>
     */
    private function createTilesWithVipsDzsave($pngPath, $pageIndex)
    {
        $tiles = array();

        $root = sprintf('%s/page_%d_dz', $this->imageDir, $pageIndex);
        $dzi  = $root . '.dzi';
        $dir  = $root . '_files';

        // Clean old outputs if any
        if (file_exists($dzi)) {
            @unlink($dzi);
        }
        if (is_dir($dir)) {
            foreach (glob($dir . '/*', GLOB_ONLYDIR) as $sub) {
                foreach (glob($sub . '/*') as $file) {
                    @unlink($file);
                }
                @rmdir($sub);
            }
            @rmdir($dir);
        }

        // Build docker + vips dzsave command
        $cmdParts = $this->makeDockerBaseCommand();

        $cmdParts[] = 'vips';
        $cmdParts[] = 'dzsave';
        $cmdParts[] = escapeshellarg($this->outputImageInContainer($pngPath));
        $cmdParts[] = escapeshellarg($this->outputImageInContainer($root));
        $cmdParts[] = '--tile-size=' . (int)$this->vipsTileSize;
        $cmdParts[] = '--overlap='   . (int)$this->tileOverlap;
        $cmdParts[] = '--depth=one';        // only highest zoom level
        $cmdParts[] = '--suffix=.png';

        $cmd = implode(' ', $cmdParts) . ' 2>/dev/null';
        $this->logInfo("Docker VIPS dzsave command: {$cmd}");
        shell_exec($cmd);

        if (!is_dir($dir)) {
            // dzsave failed or produced something unexpected
            $this->logError("Docker VIPS dzsave failed or produced unexpected output for page {$pageIndex}.");
            return array();
        }

        // Find the highest level directory (DeepZoom levels are numeric)
        $levelDirs = glob($dir . '/*', GLOB_ONLYDIR);
        if (!$levelDirs) {
            return array();
        }

        $maxLevel    = -1;
        $maxLevelDir = null;

        foreach ($levelDirs as $levelDir) {
            $base  = basename($levelDir);
            $level = (int)$base;
            if ($level > $maxLevel) {
                $maxLevel    = $level;
                $maxLevelDir = $levelDir;
            }
        }

        if ($maxLevelDir === null) {
            return array();
        }

        // Each tile: "<col>_<row>.png"
        $tileFiles = glob($maxLevelDir . '/*.png');
        if (!$tileFiles) {
            return array();
        }

        foreach ($tileFiles as $path) {
            $base = basename($path, '.png');
            $parts = explode('_', $base);
            if (count($parts) !== 2) {
                continue;
            }

            $col = (int)$parts[0];
            $row = (int)$parts[1];

            // offsets should account for overlap between tiles
            $overlap = (int)$this->tileOverlap;

            $offsetX = $col * (int)$this->vipsTileSize;
            $offsetY = $row * (int)$this->vipsTileSize;

            if ($col > 0) $offsetX -= $overlap;
            if ($row > 0) $offsetY -= $overlap;

            $offsetX = max(0, $offsetX);
            $offsetY = max(0, $offsetY);

            $tiles[] = array(
                'path'     => $path,
                'offset_x' => $offsetX,
                'offset_y' => $offsetY,
            );
        }

        return $tiles;
    }


    /**
     * Cleanup any temporary files created by this provider.
     * Safe to call multiple times.
     *
     * @return void
     */
    public function cleanup()
    {
        $files = glob($this->imageDir . '/vips_page_*.png');
        if (is_array($files)) {
            foreach ($files as $file) {
                @unlink($file); 
            }
        }
        @rmdir($this->imageDir);
    }
}
