<?php

namespace boru\ocr\Tesseract;

use boru\boruai\BoruAI;
use boru\ocr\OCRLogger;

/**
 * Lightweight DTO for common tesseract settings.
 * Keep this small; you can extend over time.
 */
class TesseractOptions
{
    /** @var string|null */
    public $lang = null; // e.g. 'eng'

    /** @var int|null */
    public $psm = null;  // Page segmentation mode

    /** @var int|null */
    public $oem = null;  // OCR engine mode

    /** @var array */
    public $config = array(); // key => value OR raw flags (see buildArgs)

    /** @var int|null */
    public $timeoutSeconds = null; // optional, used if you later swap runner

    /** @var bool */
    public $quiet = true;

    /** @var OCRLogger|null */
    public $logger = null;

        /** @var string|false */
    public $saveImages = false;

    /**
     * Docker mode: if true, run Tesseract inside a docker container.
     * The CLI runner will derive the mount point from the realpath() of the image.
     *
     * @var bool
     */
    public $dockerMode = false;

    /**
     * Docker image to use when dockerMode is true. Example: 'franky1/tesseract'.
     *
     * @var string
     */
    public $dockerImage = 'franky1/tesseract';

    /**
     * Additional docker run options (e.g. '--rm', '--network=none').
     * These will be appended before the image name.
     *
     * @var string
     */
    public $dockerOptions = '--rm';

    /**
     * Path inside the container where the host image directory is mounted.
     * Default '/tmp'.
     *
     * @var string
     */
    public $dockerMountTarget = '/tmp';


    /**
     * @param array $data
     */
    public function __construct(array $data = array())
    {
        if (isset($data['lang'])) $this->lang = $data['lang'];
        if (isset($data['psm']))  $this->psm  = $data['psm'];
        if (isset($data['oem']))  $this->oem  = $data['oem'];
        if (isset($data['config']) && is_array($data['config'])) $this->config = $data['config'];
        if (isset($data['timeoutSeconds'])) $this->timeoutSeconds = $data['timeoutSeconds'];
        if (isset($data['quiet'])) $this->quiet = (bool)$data['quiet'];
        if (isset($data['logger']) && $data['logger'] instanceof OCRLogger) {
            $this->logger = $data['logger'];
        }
        if (isset($data['saveImages'])) $this->saveImages = $data['saveImages'];
        if (isset($data['dockerMode'])) {
            $this->dockerMode = (bool)$data['dockerMode'];
            
            if (isset($data['dockerImage']) && is_string($data['dockerImage']) && $data['dockerImage'] !== '') {
                $this->dockerImage = $data['dockerImage'];
            }
            if (isset($data['dockerOptions']) && is_string($data['dockerOptions'])) {
                $this->dockerOptions = $data['dockerOptions'];
            }
            if (isset($data['dockerMountTarget']) && is_string($data['dockerMountTarget']) && $data['dockerMountTarget'] !== '') {
                $this->dockerMountTarget = $data['dockerMountTarget'];
            }
        } else {
            if(class_exists("\\boru\\boruai\\BoruAI")) {
                // If BoruAI is initialized, we can use it for docker execution if desired
                $tessConfig = BoruAI::config("tesseract", false);
                if(is_array($tessConfig)) {
                    if (isset($tessConfig['dockerImage']) && is_string($tessConfig['dockerImage']) && $tessConfig['dockerImage'] !== '') {
                        $this->dockerImage = $tessConfig['dockerImage'];
                    }
                    if (isset($tessConfig['dockerOptions']) && is_string($tessConfig['dockerOptions'])) {
                        $this->dockerOptions = $tessConfig['dockerOptions'];
                    }
                    if (isset($tessConfig['dockerMountTarget']) && is_string($tessConfig['dockerMountTarget']) && $tessConfig['dockerMountTarget'] !== '') {
                        $this->dockerMountTarget = $tessConfig['dockerMountTarget'];
                    }
                    if (isset($tessConfig['dockerMode'])) {
                        $this->dockerMode = (bool)$tessConfig['dockerMode'];
                    }   
                }
            }
        }
    }

    public function toArray()
    {
        return [
            'lang' => $this->lang,
            'psm' => $this->psm,
            'oem' => $this->oem,
            'config' => $this->config,
            'timeoutSeconds' => $this->timeoutSeconds,
            'quiet' => $this->quiet,
            'saveImages' => $this->saveImages,
            'dockerMode' => $this->dockerMode,
            'dockerImage' => $this->dockerImage,
            'dockerOptions' => $this->dockerOptions,
            'dockerMountTarget' => $this->dockerMountTarget,
        ];
    }
}
