<?php

namespace boru\ocr\Format;

class FrameFormatter
{
    /**
     * @param array $pages Array of page strings
     * @param string $title
     * @return string
     */
    public static function framePages(array $pages, $title = 'DOCUMENT OCR OUTPUT')
    {
        $out = '';
        $out .= '[BEGIN ' . $title . ']' . "\n";

        $i = 0;
        foreach ($pages as $pageText) {
            $i++;
            $out .= '[BEGIN PAGE ' . $i . ']' . "\n";
            $out .= rtrim((string)$pageText) . "\n";
            $out .= '[END PAGE ' . $i . ']' . "\n";
        }

        $out .= '[END ' . $title . ']' . "\n";
        return $out;
    }
    
    /**
     * Frame structured table candidates for model consumption.
     *
     * @param array $candidates
     * @param int $maxCandidates
     * @return string
     */
    public static function frameTableCandidates(array $candidates, $maxCandidates = 40, $title = false)
    {
        // Normalize and cap (keep most promising first)
        $rows = array();
        foreach ($candidates as $c) {
            if (!is_array($c)) continue;

            $score = isset($c['score']) ? (float)$c['score'] : 0.0;
            $kind  = isset($c['kind']) ? (string)$c['kind'] : 'unknown';

            // Ensure bbox fields exist in a consistent form
            $bbox = null;
            if (isset($c['bbox']) && is_array($c['bbox'])) {
                $bbox = $c['bbox'];
            } else {
                // fallback if your candidates store min/max bounds
                if (isset($c['minLeft']) || isset($c['maxRight']) || isset($c['minTop']) || isset($c['maxBottom'])) {
                    $minLeft = isset($c['minLeft']) ? (int)$c['minLeft'] : 0;
                    $maxRight = isset($c['maxRight']) ? (int)$c['maxRight'] : 0;
                    $minTop = isset($c['minTop']) ? (int)$c['minTop'] : 0;
                    $maxBottom = isset($c['maxBottom']) ? (int)$c['maxBottom'] : 0;
                    $bbox = array(
                        'minLeft' => $minLeft,
                        'minTop' => $minTop,
                        'maxRight' => $maxRight,
                        'maxBottom' => $maxBottom,
                    );
                }
            }

            $rows[] = array(
                'page' => isset($c['page']) ? (int)$c['page'] : null,
                'profile' => isset($c['profile']) ? (string)$c['profile'] : null,
                'region' => isset($c['region']) ? (string)$c['region'] : null,
                'kind' => $kind,
                'sourceKind' => isset($c['sourceKind']) ? (string)$c['sourceKind'] : null,
                'score' => $score,
                'cols' => isset($c['cols']) ? (int)$c['cols'] : null,
                'tokens' => isset($c['tokens']) ? (int)$c['tokens'] : null,
                'bbox' => $bbox,
                // keep light features only (avoid token bloat)
                'features' => isset($c['features']) && is_array($c['features']) ? $c['features'] : null,
            );
        }

        usort($rows, function($a, $b) {
            $as = isset($a['score']) ? (float)$a['score'] : 0.0;
            $bs = isset($b['score']) ? (float)$b['score'] : 0.0;
            if ($as == $bs) return 0;
            return ($as > $bs) ? -1 : 1;
        });

        if (count($rows) > $maxCandidates) {
            $rows = array_slice($rows, 0, $maxCandidates);
        }

        $payload = array(
            'candidateCount' => count($rows),
            'candidates' => $rows,
        );

        return ($title ? $title . "\n" : "") . json_encode($payload);
    }

}