<?php
namespace boru\ocr\Agent;

use boru\boruai\Models\File;
use boru\boruai\Models\Response;
use boru\ocr\OCRLogger;
use boru\ocr\Traits\OcrLogTrait;

class OCRAgent {
    public static $maxRetries = 3;
    public static $retryDelay = 5;

    private static $fileIDS = [];

    private $reference = null;
    private $pdfFilePath = null;
    private $fileId = null;
    private $fileType = "input_file";

    private $model = "gpt-5.1"; // Default model

    private $retries = 0;

    private $instructions = "";
    private $message = "";

    private $tools = [];

    private $chat;

    /** @var OCRLogger|null */
    protected $logger;
    use OcrLogTrait;

    public static $contextFileTypes = ["art", "bat", "brf", "c", "cls", "css", "diff", "eml", "es", "h", "hs", "htm", "html", "ics", "ifb", "java", "js", "json", "ksh", "ltx", "mail", "markdown", "md", "mht", "mhtml", "mjs", "nws", "patch", "pdf", "pl", "pm", "pot", "py", "rst", "scala", "sh", "shtml", "srt", "sty", "tex", "text", "txt", "vcf", "vtt", "xml", "yaml", "yml"];
    public static $imageFileTypes = ["bmp", "gif", "jpg", "jpeg", "png", "svg", "tiff", "webp"];

    /**
     * OCRProcessor constructor.
     * @param string $pdfPath Path to the PDF file to be processed.
     * @param AgentOptions|array $options Options for the OCR process, including model and reference.
     * @param OCRLogger|null $logger Optional logger callback.
     */
    public function __construct($pdfPath,$options = null, $logger = null) {
        $options = AgentOptions::fromInput($options);
        if($options->model()) {
            $this->model = $options->model();
        }
        if($options->reference()) {
            $this->reference = $options->reference();
        }
        if($options->logger() && is_callable($options->logger())) {
            $this->logger = $options->logger();
        }
        if($logger && $logger instanceof OCRLogger) {
            $this->logger = $logger;
        }
        $this->pdfFilePath = $pdfPath;
        //decide the file type
        if(in_array(strtolower(pathinfo($pdfPath, PATHINFO_EXTENSION)), static::$imageFileTypes)) {
            $this->fileType = "image";
        } elseif(in_array(strtolower(pathinfo($pdfPath, PATHINFO_EXTENSION)), static::$contextFileTypes)) {
            $this->fileType = "file";
        }
        //ensure the file exists
        if(!is_file($pdfPath)) {
            throw new \Exception("file does not exist: ".$pdfPath);
        }
    }

    public function init($model=null) {
        if($model === null) {
            $model = $this->model;
        }
        $this->chat = new Response();
        if($this->reference) {
            $this->chat->reference($this->reference);
        }
        $this->chat->model($model);
        $this->chat->instructions($this->instructions());
        if(!empty($this->tools())) {
            $this->chat->addTools($this->tools());
        }
    }

    public function instructions($instructions=null) {
        if($instructions !== null) {
            $this->instructions = $instructions;
        }
        return $this->instructions;
    }

    public function addMessage($message) {
        if(!is_array($this->message)) {
            if(!empty($this->message)) {
                $this->message = [$this->message];
            } else {
                $this->message = [];
            }
        }
        if(!empty($message) && !is_array($message)) {
            $this->message[] = $message;
        }
        if(is_array($message)) {
            foreach($message as $msg) {
                if(!empty($msg)) {
                    $this->message[] = $msg;
                }
            }
        } else {
            if(!empty($message)) {
                $this->message[] = $message;
            }
        }
        return $this->message;
    }

    public function message($message=null) {
        if($message !== null) {
            $this->message = $message;
        }
        return $this->message;
    }

    public function tools($tools=null) {
        if($tools !== null) {
            if(!is_array($tools)) {
                $tools = [$tools];
            }
            $this->tools = $tools;
        }
        return $this->tools;
    }

    public function fileId() {
        if($this->fileId) {
            return $this->fileId;
        }
        if(isset(static::$fileIDS[$this->pdfFilePath])) {
            $this->fileId = static::$fileIDS[$this->pdfFilePath];
            return $this->fileId;
        }
        $fileIdFile = $this->pdfFilePath . ".fileid";
        if(file_exists($fileIdFile)) {
            $this->fileId = trim(file_get_contents($fileIdFile));
            static::$fileIDS[$this->pdfFilePath] = $this->fileId;
            return $this->fileId;
        }
        $file = File::uploadFile($this->pdfFilePath, "user_data");
        $this->logInfo("Uploaded file: ".$this->pdfFilePath);
        if(!$file) {
            throw new \Exception("Failed to upload file: ".$this->pdfFilePath);
        }
        $this->fileId = $file->fileId();
        static::$fileIDS[$this->pdfFilePath] = $this->fileId;
        file_put_contents($fileIdFile, $this->fileId);
        return $file->fileId();
    }

    private function reTryOrReturn($result) {
        if($this->retries < static::$maxRetries) {
            $this->retries++;
            $this->logError("[OCR] Retrying OCR process for file: ".$this->pdfFilePath." (Attempt ".$this->retries.")");
            sleep(static::$retryDelay);
            return $this->run();
        } else {
            $this->logError("[OCR] Max retries reached for file: ".$this->pdfFilePath);
            return $result;
        }
    }

    public function run($reference=null) {
        if($reference !== null) {
            $this->reference = $reference;
            
        }
        $this->init();
        $result = null;
        try {
            $result = $this->runExecute();
        } catch (\Exception $e) {
            $this->logError("process failed with error: ".$e->getMessage());
            return $this->reTryOrReturn("[OCR FAILED - EXCEPTION: ".$e->getMessage()."]");
        }
        if($result) {
            return $result;
        } else {
            $this->logError("process failed");
            return $this->reTryOrReturn("[OCR FAILED - NO RESULT]");
        }
    }

    private function runExecute() {
        $this->init();
        if(!empty($this->message)) {
            if(is_array($this->message)) {
                foreach($this->message as $msg) {
                    $this->chat->addMessage($msg);
                }
            } else {
                $this->chat->addMessage($this->message);
            }
        }
        if($this->fileType == "file") {
            $this->chat->addFile($this->fileId());
        } elseif($this->fileType == "image") {
            $this->chat->addImage($this->fileId());
        }
        $result = $this->chat->create(null, function($streamEvent) {
        }, true);
        if($result) {
            $this->logLog("process completed");
            return $result->getResult();
        } else {
            $this->logLog("No result returned from OCR process");
            return "[OCR FAILED - NO RESULT]";
        }
    }
}