<?php
namespace boru\openai\api;

use boru\openai\api\responses\ChatResponse;
use boru\openai\OpenAI;

class Chat {
    private $model = "gpt-3.5-turbo";
    private $messages = [];

    /**
     * @var float|null
     * What sampling temperature to use, between 0 and 2. Higher values like 0.8 will make the output more random, while lower values like 0.2 will make it more focused and deterministic.
     * We generally recommend altering this or top_p but not both.
     */
    private $temperature = null; // 1

    /**
     * @var float|null
     * An alternative to sampling with temperature, called nucleus sampling, where the model considers the results of the tokens with top_p probability mass. So 0.1 means only the tokens comprising the top 10% probability mass are considered.
     * We generally recommend altering this or temperature but not both.
     */
    private $top_p = null; // 1

    /**
     * @var int|null
     * The maximum number of tokens to generate. The default is 64, and the maximum is 2048.
     */
    private $max_tokens = null; // 100

    /**
     * @var int
     * The number of candidates to return. We generally recommend this to be 1, but it can be set to any integer value.
     */
    private $n = 1;

    /**
     * @var bool
     * Whether to stream back partial progress. This is useful for long-running requests, such as generating an entire article, where you want to see progress as it's being generated.
     */
    private $stream = false;

    /**
     * @var array
     * A list of tokens which will cause the API to stop generating further tokens. This is useful if you want to control the length of the output.
     */
    private $stop = [];

    /**
     * @var float
     * A floating-point value that penalizes new tokens based on whether they appear in the text so far. Increasing this value makes the model more likely to talk about new topics, and decreasing it makes the model more likely to repeat the same line verbatim.
     */
    private $presence_penalty = 0;

    /**
     * @var float
     * A floating-point value that penalizes new tokens based on their existing frequency in the text so far. Increasing this value makes the model more likely to talk about new topics, and decreasing it makes the model more likely to repeat the same line verbatim.
     */
    private $frequency_penalty = 0;

    public function __construct($model="gpt-3.5-turbo") {
        $this->model = $model;
    }

    /**
     * @param array $parameters
     * @return array|ApiResponse
     * @throws \Exception
     */
    public static function create($parameters,$returnAsArray=false) {
        if($returnAsArray) {
            return OpenAI::request("post","chat/completions",$parameters,$returnAsArray);
        } else {
            $request = OpenAI::post("chat/completions",$parameters,$returnAsArray);
            return new ChatResponse($request->send());
        }
    }

    /**
     * @return  array|ApiResponse|false
     */
    public function send($returnAsArray=false) {
        $parameters["model"] = $this->model;
        if(!is_null($this->temperature)) {
            $parameters["temperature"] = $this->temperature;
        }
        if(!is_null($this->top_p)) {
            $parameters["top_p"] = $this->top_p;
        }
        if(!is_null($this->max_tokens)) {
            $parameters["max_tokens"] = $this->max_tokens;
        }
        $parameters["n"] = $this->n;
        $parameters["stream"] = $this->stream;
        if(!empty($this->stop)) {
            $parameters["stop"] = $this->stop;
        }
        $parameters["presence_penalty"] = $this->presence_penalty;
        $parameters["frequency_penalty"] = $this->frequency_penalty;
        $parameters["messages"] = $this->messages;
        return self::create($parameters,$returnAsArray);
    }

    /**
     * Set the model to use for the chat
     */ 
    public function setModel($model) {
        $this->model = $model;
    }
    public function setTemperature($temperature) {
        $this->temperature = $temperature;
    }
    public function setTopP($top_p) {
        $this->top_p = $top_p;
    }
    public function setMaxTokens($max_tokens) {
        $this->max_tokens = $max_tokens;
    }
    public function setN($n) {
        $this->n = $n;
    }
    public function setStream($stream) {
        $this->stream = $stream;
    }
    public function setStop($stop) {
        $this->stop = $stop;
    }
    public function setPresencePenalty($presence_penalty) {
        $this->presence_penalty = $presence_penalty;
    }
    public function setFrequencyPenalty($frequency_penalty) {
        $this->frequency_penalty = $frequency_penalty;
    }

    public function addMessage($role,$content,$name="") {
        $msg = [
            "role" => $role,
            "content" => $content,
        ];
        if($name) {
            $msg["name"] = $name;
        }
        $this->messages[] = $msg;
    }

    public static function fromArray($options=[]) {
        $chat = new Chat();
        if(isset($options["model"])) {
            $chat->setModel($options["model"]);
        }
        if(isset($options["temperature"])) {
            $chat->setTemperature($options["temperature"]);
        }
        if(isset($options["top_p"])) {
            $chat->setTopP($options["top_p"]);
        }
        if(isset($options["max_tokens"])) {
            $chat->setMaxTokens($options["max_tokens"]);
        }
        if(isset($options["n"])) {
            $chat->setN($options["n"]);
        }
        if(isset($options["stream"])) {
            $chat->setStream($options["stream"]);
        }
        if(isset($options["stop"])) {
            $chat->setStop($options["stop"]);
        }
        if(isset($options["presence_penalty"])) {
            $chat->setPresencePenalty($options["presence_penalty"]);
        }
        if(isset($options["frequency_penalty"])) {
            $chat->setFrequencyPenalty($options["frequency_penalty"]);
        }
        if(isset($options["messages"])) {
            foreach($options["messages"] as $msg) {
                $chat->addMessage($msg["role"],$msg["content"],isset($msg["name"]) ? $msg["name"] : "");
            }
        }
        return $chat;
    }
}