<?php
namespace boru\openai;

use boru\openai\api\Chat;
use boru\openai\api\endpoints\Models;
use boru\openai\api\responses\ApiBaseResponse;
use boru\openai\api\responses\ApiListResponse;


use boru\dhttp\core\Options;
use boru\dhttp\core\Request;
use boru\dhttp\core\Response;
use Exception;

class OpenAI {
    /**
     * @var \boru\dhutils\dhHttp
     */
    private static $http;
    private static $api_key;
    private static $api_version = "v1";
    private static $api_url = "https://api.openai.com";
    //private static $api_url = "http://localhost/post.php?";

    private static $isInit = false;

    private static $debugHttp = true;

    private static $contentType = "application/json";

    public static function init($api_key,$api_version="v1") {
        static::$isInit=true;
        static::setApiKey($api_key);
        static::setApiVersion($api_version);
        static::newHttp();
    }

    public static function isInit() {
        return static::$isInit;
    }
    public static function verifyInit() {
        if(!static::isInit()) {
            throw new \Exception("OpenAI is not initialized. Please call OpenAI::init(\$api_key,\$api_version) first.");
        }
    }

    /**
     * Set debug mode for HTTP requests
     * @param bool $debug
     */
    public static function debugHttp($debug=true) {
        static::$debugHttp = $debug;
    }

    public static function newHttp() {
        static::$http = new \boru\dhutils\dhHttp();
    }
    public static function setApiKey($api_key) {
        static::$api_key = $api_key;
    }
    public static function setApiVersion($api_version) {
        static::$api_version = $api_version;
    }
    public static function setApiUrl($api_url) {
        static::$api_url = $api_url;
    }

    public static function getEndpoint($path="") {
        return static::$api_url . "/" . static::$api_version . "/" . $path;
    }

    public static function setContentType($contentType) {
        static::$contentType = $contentType;
    }
    public static function resetContentType() {
        static::$contentType = "application/json";
    }

    public static function guzzleHeaders($contentType="application/json") {
        $headers = [
            "Authorization"=>"Bearer " . static::$api_key,
            "Accept"=>"application/json",
            "User-Agent"=>"OpenAI-PHP/0.1",
            "OpenAI-Source"=>"boru/openai-php",
            "OpenAI-Beta"=>"assistants=v1"
        ];
        return $headers;
    }

    private static function authHeaders(&$req) {
        $req->header("Authorization","Bearer " . static::$api_key);
        $req->header("Content-Type",static::$contentType);
        $req->header("Accept","application/json");
        $req->header("User-Agent","OpenAI-PHP/0.1");
        $req->header("OpenAI-Source","boru/openai-php");
        $req->header("OpenAI-Beta","assistants=v1");
    }

    /**
     * @param mixed $type 
     * @param string $path 
     * @param array $parameters 
     * @return array|ApiBaseResponse|false 
     * @throws Exception 
     */
    public static function request($type,$path="",$parameters=[]) {
        static::verifyInit();
        $request = new Request();
        $request->method($type);
        $request->url(static::getEndpoint($path));
        if(!empty($parameters) && is_array($parameters)) {
            $request->json($parameters);
        }
        static::authHeaders($request);
        return ApiBaseResponse::fromResponse($request->send());
    }
    /**
     * @param mixed $type 
     * @param string $path 
     * @param array $parameters 
     * @return array|ApiListResponse|false 
     * @throws Exception 
     */
    public static function requestList($type,$path="",$parameters=[],$className) {
        static::verifyInit();
        $request = new Request();
        $request->method($type);
        $request->url(static::getEndpoint($path));
        if(!empty($parameters) && is_array($parameters)) {
            $request->json($parameters);
        }
        static::authHeaders($request);
        return ApiListResponse::fromListResponse($request->send(),$className,[static::class,"requestList"],[$type,$path,$parameters,$className]);
    }

    /**
     * @param string $path
     * @param array $paramaters
     * @return Request
     * @throws \Exception
     */
    public static function post($path="",$parameters=[]) {
        static::verifyInit();
        $request = new Request();
        $request->method("post");
        $request->url(static::getEndpoint($path));
        if(!empty($parameters) && is_array($parameters)) {
            $request->json($parameters);
        }
        static::authHeaders($request);
        return $request;
    }
    /**
     * @param string $path
     * @param array $paramaters
     * @return Request
     * @throws \Exception
     */
    public static function upload($path="",$paramaters=[]) {
        static::verifyInit();
        $url = static::getEndpoint($path);
        $options = [];
        $options["headers"] = static::guzzleHeaders("multipart/form-data");
        foreach($paramaters as $key=>$value) {
            $options["multipart"][] = [
                "name"=>$key,
                "contents"=>$value
            ];
        }
        $client = new \GuzzleHttp\Client();
        $response = $client->request("POST",$url,$options);
        $body = $response->getBody()->getContents();
        return ApiBaseResponse::fromArray($body);
    }

    /**
     * @param string $path
     * @param array $paramaters
     * @return Request
     * @throws \Exception
     */
    public static function get($path="",$paramaters=[]) {
        static::verifyInit();
        $request = new Request();
        $request->method("get");
        $request->url(static::getEndpoint($path));
        if(!empty($parameters) && is_array($parameters)) {
            $request->json($parameters);
        }
        static::authHeaders($request);
        return $request;
    }

    /**
     * @param string $path
     * @param array $paramaters
     * @return Request
     * @throws \Exception
     */
    public static function put($path="",$paramaters=[]) {
        static::verifyInit();
        $request = new Request();
        $request->method("put");
        $request->url(static::getEndpoint($path));
        if(!empty($parameters) && is_array($parameters)) {
            $request->json($parameters);
        }
        static::authHeaders($request);
        return $request;
    }

    /**
     * @param string $path
     * @param array $paramaters
     * @return Request
     * @throws \Exception
     */
    public static function delete($path="",$paramaters=[]) {
        static::verifyInit();
        $request = static::$http->request("delete",static::getEndpoint($path));
        static::authHeaders($request);
        if(!empty($paramaters) && is_array($paramaters)) {
            $request->body(json_encode($paramaters));
        }
        if(static::$debugHttp) {
            $request->debug(true);
        }
        return $request;
    }


    //Convinience functions
    public static function listModels() {
        return Models::listModels();
    }
    public static function getModel($model_id) {
        return Models::getModel($model_id);
    }
    public static function chat($options=[]) {
        $chat = Chat::fromArray($options);
        return $chat->send();
    }
}