<?php
namespace boru\openai;

use boru\openai\api\Chat;
use boru\openai\api\endpoints\Models;
use boru\openai\api\responses\ApiBaseResponse;
use boru\openai\api\responses\ApiListResponse;


use boru\dhttp\core\Options;
use boru\dhttp\core\Request;
use boru\dhttp\core\Response;
use boru\dhutils\dhGlobal;
use boru\openai\models\Assistant;
use boru\openai\models\ListItem;
use boru\openai\tools\OpenAIConfig;
use Exception;

class OpenAI {
    /**
     * @var \boru\dhutils\dhHttp
     */
    private static $http;
    private static $api_key;
    private static $api_version = "v1";
    private static $api_url = "https://api.openai.com";
    //private static $api_url = "http://localhost/post.php?";

    private static $isInit = false;

    private static $debugHttp = true;

    private static $contentType = "application/json";

    public static function init($api_key=null,$configFile=null) {
        static::$isInit=true;
        if($configFile === null) {
            $configFile = "config_boruai.json";
            
        }
        OpenAIConfig::init($configFile);
        if($api_key !== null) {
            static::setApiKey($api_key);
        }
        
        static::newHttp();
    }
    public static function initConfig($configFile) {
        static::init(null,$configFile);
    }
    public static function initManual($config) {
        static::$isInit=true;
        OpenAIConfig::loadManualConfig($config);
        static::newHttp();
    }

    public static function isInit() {
        return static::$isInit;
    }
    public static function verifyInit() {
        if(!static::isInit()) {
            throw new \Exception("OpenAI is not initialized. Please call OpenAI::init(\$api_key,\$api_version) first.");
        }
    }

    /**
     * Set debug mode for HTTP requests
     * @param bool $debug
     */
    public static function debugHttp($debug=true) {
        static::$debugHttp = $debug;
    }

    public static function newHttp() {
        static::$http = new \boru\dhutils\dhHttp();
    }
    public static function setApiKey($api_key) {
        OpenAIConfig::set("openai.api_key",$api_key);
    }
    public static function setApiVersion($api_version) {
        static::$api_version = $api_version;
    }
    public static function setApiUrl($api_url) {
        static::$api_url = $api_url;
    }

    public static function getEndpoint($path="") {
        return static::$api_url . "/" . static::$api_version . "/" . $path;
    }

    public static function setContentType($contentType) {
        static::$contentType = $contentType;
    }
    public static function resetContentType() {
        static::$contentType = "application/json";
    }

    public static function guzzleHeaders($contentType="application/json") {
        $headers = [
            "Authorization"=>"Bearer " . OpenAIConfig::get("openai.api_key"),
            "Accept"=>"application/json",
            "User-Agent"=>"OpenAI-PHP/0.1",
            "OpenAI-Source"=>"boru/openai-php",
            "OpenAI-Beta"=>"assistants=v2"
        ];
        return $headers;
    }

    private static function authHeaders(&$req) {
        $req->header("Authorization","Bearer " . OpenAIConfig::get("openai.api_key"));
        $req->header("Content-Type",static::$contentType);
        $req->header("Accept","application/json");
        $req->header("User-Agent","OpenAI-PHP/0.1");
        $req->header("OpenAI-Source","boru/openai-php");
        $req->header("OpenAI-Beta","assistants=v2");
    }

    /**
     * @param mixed $type 
     * @param string $path 
     * @param array $parameters 
     * @return array|ApiBaseResponse|false 
     * @throws Exception 
     */
    public static function request($type,$path="",$parameters=[]) {
        static::verifyInit();
        $request = new Request();
        $request->method($type);
        $request->url(static::getEndpoint($path));
        if(!empty($parameters) && is_array($parameters)) {
            $request->json($parameters);
        }
        static::authHeaders($request);
        return ApiBaseResponse::fromResponse($request->send());
    }
    /**
     * @param mixed $type 
     * @param string $path 
     * @param array $parameters 
     * @return array|ApiBaseResponse|false 
     * @throws Exception 
     */
    public static function rawRequest($type,$path="",$parameters=[]) {
        static::verifyInit();
        $request = new Request();
        $request->method($type);
        $request->url(static::getEndpoint($path));
        if(!empty($parameters) && is_array($parameters)) {
            $request->json($parameters);
        }
        static::verifyInit();
        $url = static::getEndpoint($path);
        $options = [];
        $options["headers"] = static::guzzleHeaders();
        if(!empty($parameters) && is_array($parameters)) {
            $options["json"] = $parameters;
        }
        $client = new \GuzzleHttp\Client();
        $response = $client->request("POST",$url,$options);
        return $response->getBody()->getContents();
        //return ApiBaseResponse::fromArray($body);
    }
    /**
     * @param mixed $type 
     * @param string $path 
     * @param array $parameters 
     * @return array|ApiListResponse|false 
     * @throws Exception 
     */
    public static function requestList($type,$path="",$parameters=[],$className=null) {
        static::verifyInit();
        $request = new Request();
        $request->method($type);
        $request->url(static::getEndpoint($path));
        if(!empty($parameters) && is_array($parameters)) {
            $request->json($parameters);
        }
        static::authHeaders($request);
        if($className === null) {
            $className = ListItem::class;
        }
        return ApiListResponse::fromListResponse($request->send(),$className,[static::class,"requestList"],[$type,$path,$parameters,$className]);
    }

    /**
     * @param string $path
     * @param array $paramaters
     * @return Request
     * @throws \Exception
     */
    public static function post($path="",$parameters=[]) {
        static::verifyInit();
        $request = new Request();
        $request->method("post");
        $request->url(static::getEndpoint($path));
        if(!empty($parameters) && is_array($parameters)) {
            $request->json($parameters);
        }
        static::authHeaders($request);
        return $request;
    }
    /**
     * @param string $path
     * @param array $paramaters
     * @return Request
     * @throws \Exception
     */
    public static function upload($path="",$paramaters=[]) {
        static::verifyInit();
        $url = static::getEndpoint($path);
        $options = [];
        $options["headers"] = static::guzzleHeaders("multipart/form-data");
        foreach($paramaters as $key=>$value) {
            $options["multipart"][] = [
                "name"=>$key,
                "contents"=>$value
            ];
        }
        $client = new \GuzzleHttp\Client();
        $response = $client->request("POST",$url,$options);
        $body = $response->getBody()->getContents();
        return ApiBaseResponse::fromArray($body);
    }

    /**
     * @param string $path
     * @param array $paramaters
     * @return Request
     * @throws \Exception
     */
    public static function get($path="",$paramaters=[]) {
        static::verifyInit();
        $request = new Request();
        $request->method("get");
        $request->url(static::getEndpoint($path));
        if(!empty($parameters) && is_array($parameters)) {
            $request->json($parameters);
        }
        static::authHeaders($request);
        return $request;
    }

    /**
     * @param string $path
     * @param array $paramaters
     * @return Request
     * @throws \Exception
     */
    public static function put($path="",$paramaters=[]) {
        static::verifyInit();
        $request = new Request();
        $request->method("put");
        $request->url(static::getEndpoint($path));
        if(!empty($parameters) && is_array($parameters)) {
            $request->json($parameters);
        }
        static::authHeaders($request);
        return $request;
    }

    /**
     * @param string $path
     * @param array $paramaters
     * @return Request
     * @throws \Exception
     */
    public static function delete($path="",$paramaters=[]) {
        static::verifyInit();
        $request = static::$http->request("delete",static::getEndpoint($path));
        static::authHeaders($request);
        if(!empty($paramaters) && is_array($paramaters)) {
            $request->body(json_encode($paramaters));
        }
        if(static::$debugHttp) {
            $request->debug(true);
        }
        return $request;
    }

    public static function getRelPath($from, $to, $ps = DIRECTORY_SEPARATOR)
    {
        $arFrom = explode($ps, rtrim($from, $ps));
        $arTo = explode($ps, rtrim($to, $ps));
        while(count($arFrom) && count($arTo) && ($arFrom[0] == $arTo[0])) {
            array_shift($arFrom);
            array_shift($arTo);
        }
        if(empty($arFrom) && count($arTo) === 1) {
            return ".".$ps.$arTo[0];
        }
        return str_pad("", count($arFrom) * 3, '..'.$ps).implode($ps, $arTo);
    }

    /**
     * @param string $path
     * @param array $paramaters
     * @return \boru\dhdb\dhDB
     * @throws \Exception
     */
    public static function db($dbconfig=null) {
        if($dbconfig) {
            return dhGlobal::db($dbconfig);
        }
        if(OpenAIConfig::get("dbconfig.use",false)) {
            return dhGlobal::db(OpenAIConfig::get("dbconfig"));
        }
        return dhGlobal::db();
    }

    public static function config($key,$default=null) {
        return OpenAIConfig::get($key,$default);
    }
    public static function setConfig($key,$value) {
        return OpenAIConfig::set($key,$value);
    }

    public static function assertConfig() {
        if(OpenAIConfig::get("openai.api_key","") == "") {
            throw new \Exception("OpenAI API Key not set");
        }
    }

    public static function assistant($name) {
        if(is_object($name) && get_class($name) == Assistant::class) {
            return $name;
        }
        if(is_array($name) && isset($name["id"])) {
            return Assistant::fromArray($name);
        }
        
        $assistantId = OpenAIConfig::get("assistants.".$name,false);
        if($assistantId !== false) {
            return Assistant::fromId($assistantId);
        }

        if(substr($name,0,5) == "asst_") {
            return Assistant::fromId($name);
        }
        return Assistant::fromName($name);
        
    }

    public static function chat($options=[]) { 
        $chat = Chat::fromArray($options); 
        return $chat->send(); 
    } 
}