<?php
namespace boru\openai\models;

use boru\dhutils\dhGlobal;
use boru\openai\OpenAI;
use boru\openai\models\Assistant;
use boru\openai\models\Content;
use boru\openai\OpenAI as OpenaiOpenAI;
use boru\openai\tools\OpenAIConfig;
use boru\openai\tools\PDFConverter;
use boru\openai\tools\Summarize;

class PDFDocument extends Base {
    private static $tableName;
    private $id;
    private $reference = "";
    private $fileName = "";
    private $filesize;
    private $pages;
    private $md5;
    private $created;
    private $userid = 0;
    private $paged;
    private $summarized;
    private $summary = "";
    private $parsed;
    private $results;

    public static function tableName($tableName=null) {
        if($tableName) {
            static::$tableName = $tableName;
        }
        if(static::$tableName === null) {
            static::$tableName = OpenAIConfig::get("tables.documents","boru_openai_documents");
        }
        return static::$tableName;
    }

    public function __construct($idOrArray=null) {
        if ($idOrArray !== null) {
            if (is_array($idOrArray)) {
                $this->loadFromArray($idOrArray);
            } else if (is_numeric($idOrArray)) {
                $this->loadFromId($idOrArray);
            }
        }
    }

    public function id() {
        return $this->id;
    }
    public function reference($reference=null) {
        if ($reference !== null) {
            $this->reference = $reference;
            return $this;
        }
        return $this->reference;
    }
    public function fileName($fileName=null) {
        if ($fileName !== null) {
            $this->fileName = $fileName;
            return $this;
        }
        return $this->fileName;
    }
    public function filesize($filesize=null) {
        if ($filesize !== null) {
            $this->filesize = $filesize;
            return $this;
        }
        return $this->filesize;
    }
    public function pages($pages=null) {
        if ($pages !== null) {
            $this->pages = $pages;
            return $this;
        }
        return $this->pages;
    }
    public function md5($md5=null) {
        if ($md5 !== null) {
            $this->md5 = $md5;
            return $this;
        }
        return $this->md5;
    }
    public function created($created=null) {
        if ($created !== null) {
            $this->created = $created;
            return $this;
        }
        return $this->created;
    }
    public function userid($userid=null) {
        if ($userid !== null) {
            $this->userid = $userid;
            return $this;
        }
        return $this->userid;
    }
    public function paged($paged=null) {
        if ($paged !== null) {
            $this->paged = $paged;
            return $this;
        }
        return $this->paged;
    }
    public function summarized($summarized=null) {
        if ($summarized !== null) {
            $this->summarized = $summarized;
            return $this;
        }
        return $this->summarized;
    }
    public function summary($summary=null) {
        if ($summary !== null) {
            $this->summary = $summary;
            return $this;
        }
        return $this->summary;
    }
    public function parsed($parsed=null) {
        if ($parsed !== null) {
            $this->parsed = $parsed;
            return $this;
        }
        return $this->parsed;
    }
    public function results($results=null) {
        if ($results !== null) {
            $this->results = $results;
            return $this;
        }
        return $this->results;
    }
    public function resultsAsJson($pretty=true) {
        if($pretty) {
            return json_encode($this->results,JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES);
        }
        return json_encode($this->results);
    }
    public function isPaged() {
        return !empty($this->paged);
    }
    public function isSummarized() {
        return !empty($this->summarized);
    }
    public function isParsed() {
        return !empty($this->parsed);
    }

    public function calcStats($save=true) {
        if(file_exists($this->fileName)) {
            $this->filesize = filesize($this->fileName);
            $this->md5 = md5_file($this->fileName);
            if($save) {
                $this->save();
            }
        }
        return $this;
    }

    public function ocr($force=false) {
        $pages = $this->getPages();
        if(count($pages) == 0) {
            $this->upload();
            $pages = $this->getPages();
        }
        foreach($pages as $page) {
            $page->ocr($force);
        }

    }


    public function upload($force=false,$ocr=false) {
        if(!$force && !empty($this->getPages())) {
            return $this;
        }
        $this->paged = date("Y-m-d H:i:s");
        //$converter = new PDFConverter($this->fileName,OpenAIConfig::get("outputdir"));
        //$converter->convert(false);
        //$fileIds = $converter->upload();
        $openAiFiles = PDFConverter::convertPDF($this->fileName,OpenAIConfig::get("outputdir"));
        foreach($openAiFiles as $file) {
            $page = PDFPage::create($this->id,$file->id());
            $page->uploaded(date("Y-m-d H:i:s"));
            if($ocr) {
                $page->ocr();
            }
            $page->save();
        }
        $this->loadPages();
        if(count($this->getPages()) > 0) {
            $this->pages(count($this->getPages()));
        } else {
            throw new \Exception("No pages uploaded");
        }
        $this->save();
        return $this;
    }

    /** @var PDFPage[] */
    private $pagesCache = [];

    /**
     * @param string|null $field
     * @return PDFPage[]|string[]
     */
    public function getPages($field=null) {
        if (count($this->pagesCache) == 0) {
            $this->pagesCache = $this->loadPages();
        }
        if($field) {
            return array_map(function($page) use ($field) {
                return $page->$field();
            },$this->pagesCache);
        }
        return $this->pagesCache;
    }

    /**
     * @return PDFPage[]
     */
    private function loadPages() {
        $this->pagesCache = PDFPage::queryByDocId($this->id);
        return $this->pagesCache;
    }

    public function summarize($useOcr=true,$debug=false) {
        if(empty($this->getPages())) {
            $this->upload();
            if(empty($this->getPages())) {
                throw new \Exception("No pages to summarize");
                return false;
            }
        }

        $this->summarized = date("Y-m-d H:i:s");
        if($useOcr) {
            $pageIds = $this->getPages("id");
            $summarizer = new Summarize(["pageIds"=>$pageIds]);
            if($debug) {
                dhGlobal::outLine("Summarizing pages: ".implode(",",$pageIds));
            }
        } else {
            $fileIds = $this->getPages("openai_fileid");
            $summarizer = new Summarize(["fileIds"=>$fileIds]);
            if($debug) {
                dhGlobal::outLine("Summarizing pages: ".implode(",",$fileIds));
            }
        }
        $summary = $summarizer->run();
        if($debug) {
            dhGlobal::outLine("Summary: ".$summary);
        }
        $this->summary($summary);
        $this->summarized(date("Y-m-d H:i:s"));
        $this->save();
        return $this;
    }

    public function parse() {
        if(empty($this->summary)) {
            $this->summarize();
            if(empty($this->summary)) {
                return false;
            }
        }
        $parseAssistant = OpenAI::assistant("parse");
        if(!$parseAssistant) {
            throw new \Exception("No parse assistant found, please set one in the config: assistants.parse");
        }
        $content = new Content();
        $content->addText("extract the information from the provided summary of the document");
        $content->addText($this->summary);
        $parseAssistant->addMessage("user",$content);
        $this->results($parseAssistant->json(true));
        $this->parsed(date("Y-m-d H:i:s"));
        $this->save();
        return $this;
    }

    public function save() {
        global $adb;
        if(empty($this->filesize)) {
            $this->calcStats(false);
        }
        if(empty($this->md5)) {
            $this->calcStats(false);
        }
        $params = [
            $this->reference,
            $this->fileName,
            $this->userid,
            $this->paged,
            $this->summarized,
            $this->summary,
            $this->pages,
            $this->filesize,
            $this->md5,
            $this->parsed,
            json_encode($this->results)
        ];
        $db = OpenAI::db();
        if ($this->id) {
            $params[] = $this->created;
            $params[] = $this->id;
            $db->query("UPDATE ".static::tableName()." SET `reference`=?, filename=?, userid=?, paged=?, summarized=?, summary=?, `pages`=?, filesize=?, `md5`=?, `parsed`=?, `results`=?, `created`=? WHERE id=?", $params);
            exit();
        } else {
            $db->query("INSERT INTO ".static::tableName()." (`reference`, filename, userid, paged, summarized, summary, `pages`, filesize, `md5`, `parsed`, `results`, `created`) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())", $params);
            $this->id = $db->lastInsertId();
            $this->loadFromId($this->id);
        }
        return $this;
    }


    public function loadFromArray($array) {
        $this->id = $array['id'];
        $this->reference = $array['reference'];
        $this->fileName = $array['filename'];
        $this->filesize = $array['filesize'];
        $this->pages = $array['pages'];
        $this->md5 = $array['md5'];
        $this->created = $array['created'];
        $this->userid = $array['userid'];
        $this->paged = $array['paged'];
        $this->summarized = $array['summarized'];
        $this->summary = $array['summary'];
        $this->parsed = $array['parsed'];
        $this->results = empty($array['results']) ? [] : json_decode($array['results'],true);
        if(!is_array($this->results)) {
            $this->results = [];
        }
    }
    public function loadFromId($id) {
        $doc = static::queryById($id);
        $this->loadFromArray($doc);
    }
    public function toArray() {
        return [
            'id' => $this->id,
            'reference' => $this->reference,
            'filename' => $this->fileName,
            'filesize' => $this->filesize,
            'pages' => $this->pages,
            'md5' => $this->md5,
            'created' => $this->created,
            'userid' => $this->userid,
            'paged' => $this->paged,
            'summarized' => $this->summarized,
            'summary' => $this->summary,
            'parsed' => $this->parsed,
            'results' => $this->results
        ];
    }

    public static function search($where=[]) {
        global $adb;
        $whereClause = "";
        $params = [];
        foreach($where as $key => $value) {
            if($whereClause != "") {
                $whereClause .= " AND ";
            }
            $whereClause .= $key."=?";
            $params[] = $value;
        }
        $res = $adb->pquery("SELECT * FROM ".static::tableName()." WHERE ".$whereClause, $params);
        $docs = [];
        while($doc = $adb->fetchByAssoc($res,-1,false)) {
            $docs[] = new PDFDocument($doc);
        }
        return $docs;
    }
    
    public static function queryById($id) {
        $db = OpenAI::db();
        $res = $db->run("SELECT * FROM ".static::tableName()." WHERE id=?", [$id]);
        while($row = $db->next($res)) {
            return $row->asArray();
        }
        return false;
    }

    public static function getNeedsPaged() {
        $db = OpenAI::db();
        $res = $db->run("SELECT * FROM ".static::tableName()." WHERE paged IS NULL ORDER BY `created` ASC", []);
        $docs = [];
        while($row = $db->next($res)) {
            $docs[] = new PDFDocument($row->asArray());
        }
        return $docs;
    }
    public static function getNeedsSummarized() {
        $db = OpenAI::db();
        $res = $db->run("SELECT * FROM ".static::tableName()." WHERE summarized IS NULL ORDER BY `created` ASC", []);
        $docs = [];
        while($row = $db->next($res)) {
            $docs[] = new PDFDocument($row->asArray());
        }
        return $docs;
    }
    public static function getNeedsParsed() {
        $db = OpenAI::db();
        $res = $db->run("SELECT * FROM ".static::tableName()." WHERE parsed IS NULL ORDER BY `created` ASC", []);
        $docs = [];
        while($row = $db->next($res)) {
            $docs[] = new PDFDocument($row->asArray());
        }
        return $docs;
    }

    public static function exists($fileName) {
        global $adb;
        if(file_exists($fileName)) {
            $md5 = md5_file($fileName);
            $fileSize = filesize($fileName);
            $db = OpenAI::db();
            $res = $db->run("SELECT * FROM ".static::tableName()." WHERE md5=? AND filesize=?", [$md5,$fileSize]);
            while($row = $db->next($res)) {
                return $row->asArray();
            }
        }
        return false;
    }

    public static function create($fileName,$userId,$reference="",$dupeCheck=true) {
        if($dupeCheck) {
            $doc = static::exists($fileName);
            if($doc) {
                return new PDFDocument($doc);
            }
        }
        $doc = new PDFDocument();
        $doc->fileName($fileName);
        $doc->userid($userId);
        $doc->reference($reference);
        $doc->save();
        return $doc;
    }
}