<?php
namespace boru\openai\tools;

use boru\openai\api\endpoints\Assistants;
use boru\openai\api\endpoints\Files;
use boru\openai\models\Assistant;
use boru\openai\models\Content;
use boru\openai\models\File;
use boru\openai\OpenAI;

class OCR {
    private $detail = "high";
    private $fileId;
    private $file;
    private $fileUrl;
    private $assistant;
    private $deleteFile = false;
    private $prompts = [
        "before" => "",
        "after" => "",
    ];

    private static $assistantCache = [];

    public function __construct($options=[]) {
        if(!is_array($options)) {
            throw new \Exception("Options must be an array");
        }
        if(isset($options["file"])) {
            $this->file($options["file"]);
        }
        if(isset($options["fileId"])) {
            $this->fileId($options["fileId"]);
        }
        if(isset($options["fileUrl"])) {
            $this->fileUrl($options["fileUrl"]);
        }
        if(isset($options["detail"])) {
            $this->detail($options["detail"]);
        }
        if(isset($options["deleteFile"])) {
            $this->deleteFile($options["deleteFile"]);
        } elseif(isset($options["delete"])) {
            $this->deleteFile($options["delete"]);
        }
        if(isset($options["prompts"]) && is_array($options["prompts"])) {
            foreach($options["prompts"] as $key => $value) {
                $this->prompt($key,$value);
            }
        }
        if(isset($options["before"])) {
            $this->prompt("before",$options["before"]);
        }
        if(isset($options["after"])) {
            $this->prompt("after",$options["after"]);
        }
        if(isset($options["assistant"])) {
            $this->assistant($options["assistant"]);
        } else {
            $this->assistant("ocr");
        }
    }

    public function prompt($type,$data=null) {
        if(!isset($this->prompts[$type])) {
            throw new \Exception("Prompt type not found");
        }
        if($data) {
            $this->prompts[$type] = $data;
        }
        if(empty($this->prompts[$type])) {
            $this->prompts[$type] = OpenAI::config("prompts.ocr.".$type);
        }
        return $this->prompts[$type];
    }
    public function prompts($prompts=null) {
        if($prompts) {
            if(!is_array($prompts)) {
                throw new \Exception("Prompts must be an array");
            }
            $this->prompts = $prompts;
        }
        foreach($this->prompts as $key => $value) {
            $this->prompt($key,$value);
        }
        return $this->prompts;
    }
    public function deleteFile($deleteFile=null) {
        if($deleteFile!==null) {
            $this->deleteFile = $deleteFile ? true : false;
        }
        return $this->deleteFile;
    }

    public function detail($detail=null) {
        if ($detail !== null) {
            $this->detail = $detail == "high" ? "high" : "low";
            return $this;
        }
        return $this->detail == "high" ? "high" : "low";
    }
    public function highDetail() {
        return $this->detail("high");
    }
    public function lowDetail() {
        return $this->detail("low");
    }
    public function file($file=null) {
        if($file!==null) {
            if($file instanceof File) {
                $this->fileId($file->id());
            } elseif(is_string($file)) {
                $this->file = File::upload($file,"vision");
            } else {
                $this->file = $file;
            }
        }
        return $this->file;
    }
    public function fileId($fileId=null) {
        if($fileId!==null) {
            $this->fileId = $fileId;
        }
        return $this->fileId;
    }
    public function fileUrl($fileUrl=null) {
        if($fileUrl!==null) {
            $this->fileUrl = $fileUrl;
        }
        return $this->fileUrl;
    }
    public function assistant($assistant=null) {
        if($assistant!==null) {
            if($assistant instanceof Assistant) {
                $this->assistant = clone $assistant;
            } else {
                if(!isset(self::$assistantCache[$assistant])) {
                    self::$assistantCache[$assistant] = Assistant::fromInput($assistant);
                }
                $this->assistant = clone self::$assistantCache[$assistant];
            }
        }
        return $this->assistant;
    }

    /**
     * @param bool $force
     * @param Assistant|string $assistant
     * @return string
     */
    public function run($assistant=null) {
        if($assistant) {
            $this->assistant($assistant);
        }
        $assistant = $this->assistant();
        
        if(!empty($this->prompt("before"))) {
            $content = new Content();
            $content->addText($this->prompt("before"));
            $assistant->addMessage("user",$content);
        }
        $content = new Content();
        $content->addText("OCR This Image.");
        if($this->fileId()) {
            $content->addImageFile($this->fileId(),$this->detail());
        } elseif($this->fileUrl()) {
            $content->addImageUrl($this->fileUrl());
        } elseif($this->file()) {
            $content->addImageFile($this->file->id(),$this->detail());
        } else {
            throw new \Exception("fileId, fileUrl must be provided.. or a file to upload must be provided");
        }
        $assistant->addMessage("user",$content);
        if(!empty($this->prompt("after"))) {
            $content = new Content();
            $content->addText($this->prompt("after"));
            $assistant->addMessage("user",$content);
        }

        $result = $assistant->output("\n");
        if(substr($result,0,3) == "```") {
            $result = substr($result,3);
            //remove "plaintext" if it is the next characters
            if(substr($result,0,9) == "plaintext") {
                $result = substr($result,9);
            }
            //remove "markdown" if it is the next characters
            if(substr($result,0,8) == "markdown") {
                $result = substr($result,8);
            }
            //remove "json" if it is the next characters
            if(substr($result,0,4) == "json") {
                $result = substr($result,4);
            }
            if(substr($result,-3) == "```") {
                $result = substr($result,0,-3);
            }
        }
        if(substr($result,0,11) == "##SUCCESS##") {
            $result = substr($result,11);
        }
        if($this->deleteFile()) {
            if($this->file()) {
                echo "Deleting file: ".$this->file()->id()."\n";
                $this->file()->delete();
            }
        }
        return $result;
    }

    /**
     * Quick OCR of a file.. WILL DELETE the file from OpenAI after processing. If you want to keep the file, you can set the deleteFile option to false.
     * @param string $fileName
     * @param array $options an array of options to pass to the OCR constructor. Options include:
     *     - before (string) [default: ""] - text to add before the OCR prompt
     *     - after (string) [default: ""] - text to add after the OCR prompt
     *     - detail (high/low) [default: high]
     *     - deleteFile (true/false) [default: true]
     *     - assistant (Assistant object or string name of assistant) [default: ocr]
     * @return string
     */
    public static function ocrFile($fileName,$options=[]) {
        $ocr = new OCR($options);
        $ocr->file($fileName);
        if(!isset($options["deleteFile"]) && !isset($options["delete"])) {
            $ocr->deleteFile(true);
        }
        return $ocr->run();
    }

    /**
     * Quick OCR of a file that is already uploaded to OpenAI. The file will not be deleted by default. If you want to delete the file after processing, you can set the deleteFile option to true.
     * @param string $fileId
     * @param array $options an array of options to pass to the OCR constructor. Options include:
     *     - before (string) [default: ""] - text to add before the OCR prompt
     *     - after (string) [default: ""] - text to add after the OCR prompt
     *     - detail (high/low) [default: high]
     *     - deleteFile (true/false) [default: false]
     *     - assistant (Assistant object or string name of assistant) [default: ocr]
     * @return string
     */
    public static function ocrFileId($fileId,$options=[]) {
        $ocr = new OCR($options);
        $ocr->fileId($fileId);
        return $ocr->run();
    }

    /**
     * Quick OCR of a file using a publicly accessible URL. No file will be uploaded or stored on OpenAI. The deleteFile option has no effect.
     * @param string $fileUrl
     * @param array $options an array of options to pass to the OCR constructor. Options include:
     *      - before (string) [default: ""] - text to add before the OCR prompt
     *      - after (string) [default: ""] - text to add after the OCR prompt
     *      - detail (high/low) [default: high]
     *      - assistant (Assistant object or string name of assistant) [default: ocr]
     * @return string
     */
    public static function ocrFileUrl($fileUrl,$options=[]) {
        $ocr = new OCR($options);
        $ocr->fileUrl($fileUrl);
        return $ocr->run();
    }
}