<?php
namespace boru\openai\tools;

use boru\openai\api\endpoints\Assistants;
use boru\openai\api\endpoints\Files;
use boru\openai\models\Assistant;
use boru\openai\models\PDFPage;
use boru\openai\OpenAI;
use Exception;

class Summarize {

    private $prompts = [
        "prefix" => "",
        "context" => "",
        "update" => "",
        "new" => ""
    ];

    private $fileIds = [];
    private $pageIds = [];
    private $detail = "high";
    private $assistantId;
    private $assistantName;
    private $assistant;

    private $context = "";

    private $pagesPerRun = 7;

    public $debugSteps = false;
    
    /**
     * 
     * @param array $options Valid options are fileIds, assistantId, assistantName, prompts
     * @return void 
     * @throws Exception 
     */
    public function __construct($options=[]) {
        if(!is_array($options)) {
            throw new \Exception("Options must be an array");
        }
        $this->fileIds = isset($options["fileIds"]) ? $options["fileIds"] : [];
        $this->pageIds = isset($options["pageIds"]) ? $options["pageIds"] : [];
        $assistantName = isset($options["assistantName"]) ? $options["assistantName"] : null;
        if($assistantName) {
            $this->assistantName($assistantName);
        }
        $assistantId = isset($options["assistantId"]) ? $options["assistantId"] : null;
        if($assistantId) {
            $this->assistantId($assistantId);
        }
        if(!$this->assistant) {
            $assistantId = OpenAIConfig::get("assistants.summary",false);
            if($assistantId !== false) {
                $this->assistantId($assistantId);
            }
        }
        $detail = isset($options["detail"]) ? $options["detail"] : "high";
        $prompts = isset($options["prompts"]) ? $options["prompts"] : null;
        if($prompts) {
            $this->prompts($prompts);
        }
    }
    public function fileIds($fileIds=null) {
        if ($fileIds) {
            $this->fileIds = $fileIds;
        }
        return $this->fileIds;
    }
    public function pageIds($pageIds=null) {
        if ($pageIds) {
            $this->pageIds = $pageIds;
        }
        return $this->pageIds;
    }
    public function assistantName($assistantName=null) {
        if ($assistantName) {
            $assistant = Assistant::fromName($assistantName);
            if(!$assistant) {
                throw new \Exception("Assistant not found");
            }
            $this->assistantName = $assistant->name();
            $this->assistantId = $assistant->id();
            $this->assistant = $assistant;
        }
        return $this->assistantName;
    }
    public function assistantId($assistantId=null) {
        if ($assistantId) {
            $assistant = Assistant::fromId($assistantId);
            if(!$assistant) {
                throw new \Exception("Assistant not found");
            }
            $this->assistantName = $assistant->name();
            $this->assistantId = $assistant->id();
            $this->assistant = $assistant;
        }
        return $this->assistantId;
    }
    public function detail($detail=null) {
        if ($detail) {
            $this->detail = $detail == "low" ? "low" : "high";
        }
        return $this->detail == "low" ? "low" : "high";
    }
    public function highDetail() {
        return $this->detail("high");
    }
    public function lowDetail() {
        return $this->detail("low");
    }
    
    public function prompt($type,$data=null) {
        if(!isset($this->prompts[$type])) {
            throw new \Exception("Prompt type not found");
        }
        if($data) {
            $this->prompts[$type] = $data;
        }
        if(empty($this->prompts[$type])) {
            $this->prompts[$type] = OpenAI::config("prompts.summary.".$type);
        }
        return $this->prompts[$type];
    }
    public function prompts($prompts=null) {
        if($prompts) {
            if(!is_array($prompts)) {
                throw new \Exception("Prompts must be an array");
            }
            $this->prompts = $prompts;
        }
        foreach($this->prompts as $key => $value) {
            $this->prompt($key,$value);
        }
        return $this->prompts;
    }

    public function run($force=false) {
        if(count($this->pageIds) > 0) {
            return $this->runIds("page",$force);
        }
        if(count($this->fileIds) > 0) {
            return $this->runIDs("file",$force);
        }
        throw new \Exception("No pages or files to summarize");
    }
    
    private function runIDs($type="file",$force=false) {
        if($force && !empty($this->context)) {
            $this->context = "";
        }
        if(!empty($this->context)) {
            return $this->context;
        }
        if($type == "file") {
            if(count($this->fileIds) == 0) {
                throw new \Exception("No files to summarize");
            }
            $ids = $this->fileIds;
        } elseif($type == "page") {
            if(count($this->pageIds) == 0) {
                throw new \Exception("No pages to summarize");
            }
            $ids = $this->pageIds;
        } else {
            throw new \Exception("Invalid type");
        }
        
        if(!$this->assistant) {
            throw new \Exception("No assistant set");
        }
        $this->context = "";
        $donePages = 0;
        $doneIds = [];
        $totalPages = count($ids);
        foreach($ids as $i => $id) {
            $doneIds[] = $id;
            $donePages++;
            if($donePages % $this->pagesPerRun == 0 || $donePages == count($ids)) {
                $this->summarizeCall($type,$doneIds,$totalPages,$donePages);
                if($this->debugSteps) {
                    echo "Done Pages: $donePages\n";
                    echo "$this->context\n";
                    echo "-----------------------------------\n";
                }
                $doneIds = [];
            }
        }
        return $this->context;
    }

    private function summarizeCall($type="file",$ids,$totalPages,$donePages) {
        $prompt = str_replace("{TOTAL_PAGES}",$totalPages,$this->prompt("prefix"));
        $prompt = str_replace("{THIS_PAGES}",count($ids),$prompt);
        $prefix = [
            "type"=> "text",
            "text"=>$prompt,
        ];
        $content = [];
        $content[] = $prefix;
        if(!empty($this->context)) {
            $content[] = [
                "type"=>"text",
                "text"=>str_replace("{SUMMARY}",$this->context,str_replace("{DONE_PAGES}",$donePages,$this->prompt("context"))),
            ];
            $content[] = [
                "type"=>"text",
                "text"=>$this->prompt("update"),
            ];
        } else {
            $content[] = [
                "type"=>"text",
                "text"=>$this->prompt("new"),
            ];
        }
        foreach($ids as $id) {
            if($type == "file") {
                $content[] = [
                    "type"=>"image_file",
                    "image_file"=>["file_id"=>$id,"detail"=>$this->detail()],
                ];
            } elseif($type == "page") {
                $page = new PDFPage($id);
                $content[] = [
                    "type"=>"text",
                    "text"=>"PAGE CONTENT:\n\n".$page->ocr(),
                ];
            }
        }
        $assistant = clone $this->assistant;
        $assistant->addMessage("user",$content);
        $result = $assistant->run();
        //print_r($result);
        $this->context = "";
        foreach($result as $message) {
            $this->context .= $message->value()."\n";
        }

        //echo "Current Context: $currentContext\n\n";
        return $this->context;
    }
}