<?php
namespace boru\openai;

use boru\openai\api\Chat;
use boru\openai\api\endpoints\Models;
use boru\openai\api\responses\ApiBaseResponse;
use boru\openai\api\responses\ApiListResponse;


use boru\dhttp\core\Options;
use boru\dhttp\core\Request;
use boru\dhttp\core\Response;
use boru\dhutils\dhGlobal;
use boru\openai\models\Assistant;
use boru\openai\models\ListItem;
use boru\openai\OpenAIConfig;
use boru\dhdb\dhDB;
use boru\output\Output;
use Exception;

class OpenAI {
    private static $api_version = "v1";
    private static $api_url = "https://api.openai.com";
    //private static $api_url = "http://localhost/post.php?";

    private static $isInit = false;

    private static $debugMode = false;
    private static $debugHttp = false;

    private static $guzzleOptions = [
            'timeout' => 60, // Response timeout
            'connect_timeout' => 5, // Connection timeout
    ];

    public static function init($api_key=null,$configFile=null) {
        static::$isInit=true;
        if($configFile === null) {
            $configFile = "config_boruai.json";
            
        }
        OpenAIConfig::init($configFile);
        if($api_key !== null) {
            static::setApiKey($api_key);
        }
    }
    public static function initConfig($configFile) {
        static::init(null,$configFile);
    }
    public static function initManual($config) {
        static::$isInit=true;
        OpenAIConfig::loadManualConfig($config);
    }

    public static function isInit() {
        return static::$isInit;
    }
    public static function verifyInit() {
        if(!static::isInit()) {
            throw new \Exception("OpenAI is not initialized. Please call OpenAI::init(\$api_key,\$api_version) first.");
        }
    }

    /**
     * Set debug mode for HTTP requests
     * @param bool $debug
     */
    public static function debugHttp($debug=true) {
        static::$debugHttp = $debug;
    }

    public static function setApiKey($api_key) {
        OpenAIConfig::set("openai.api_key",$api_key);
    }
    public static function setApiVersion($api_version) {
        static::$api_version = $api_version;
    }
    public static function setApiUrl($api_url) {
        static::$api_url = $api_url;
    }

    public static function getEndpoint($path="") {
        return static::$api_url . "/" . static::$api_version . "/" . $path;
    }

    public static function debugMode($debug=null) {
        if($debug !== null) {
            static::$debugMode = $debug ? true : false;
        }
        return static::$debugMode ? true : false;
    }
    public static function debugHttpMode($debug=null) {
        if($debug !== null) {
            static::$debugHttp = $debug ? true : false;
        }
        return static::$debugHttp ? true : false;
    }

    public static function guzzleOptions(...$options) {
        $return = static::$guzzleOptions;
        foreach($options as $option) {
            if(is_array($option)) {
                $return = array_merge($return,$option);
            }
        }
        return $return;
    }

    public static function guzzleHeaders($contentType="application/json") {
        $headers = [
            "Authorization"=>"Bearer " . OpenAIConfig::get("openai.api_key"),
            "Accept"=>"application/json",
            "User-Agent"=>"OpenAI-PHP/0.1",
            "OpenAI-Source"=>"boru/openai-php",
            "OpenAI-Beta"=>"assistants=v2"
        ];
        return $headers;
    }

    /**
     * @param mixed $type 
     * @param string $path 
     * @param array $parameters 
     * @return ApiBaseResponse|false 
     * @throws Exception 
     */
    public static function request($type,$path="",$parameters=[],$guzzleOptions=[]) {
        static::verifyInit();

        $url = static::getEndpoint($path);
        self::printHttpDebug(strtoupper($type),"Requesting to: ".$url);
        $options = [];
        $options["headers"] = static::guzzleHeaders();
        if(!empty($parameters) && is_array($parameters)) {
            $options['json'] = $parameters;
        }
        $options = static::guzzleOptions($options,$guzzleOptions);
        $client = new \GuzzleHttp\Client();
        $i=0; $isError = false;
        while($i++<3) {
            try {
                $response = $client->request(strtoupper($type),$url,$options);
                $parsed = static::responseParse($response);
                self::printHttpDebug(strtoupper($type),"Completed");
                return ApiBaseResponse::fromArray($parsed);
            } catch(Exception $e) {
                $isError = $e;
                self::printHttpDebug(strtoupper($type),"Error with request.. retrying..");
            }
        }
        if($isError) {
            self::printHttpDebug(strtoupper($type),"Error with request.. throwing..");
            throw $isError;
        }
    }
    /**
     * @param mixed $type 
     * @param string $path 
     * @param array $parameters 
     * @return mixed
     * @throws Exception 
     */
    public static function rawRequest($type,$path="",$parameters=[]) {
        static::verifyInit();
        $url = static::getEndpoint($path);
        self::printHttpDebug(strtoupper($type),"Requesting to: ".$url);
        $options = [];
        $options["headers"] = static::guzzleHeaders();
        if(!empty($parameters) && is_array($parameters)) {
            $options["json"] = $parameters;
        }
        $client = new \GuzzleHttp\Client();
        $i=0; $isError = false;
        while($i++<3) {
            try {
                $response = $client->request(strtoupper($type),$url,$options);
                self::printHttpDebug(strtoupper($type),"Completed");
                return $response->getBody()->getContents();
            } catch(Exception $e) {
                $isError = $e;
                self::printHttpDebug(strtoupper($type),"Error with request.. retrying..");
            }
        }
        if($isError) {
            self::printHttpDebug(strtoupper($type),"Error with request.. throwing..");
            throw $isError;
        }
        //return ApiBaseResponse::fromArray($body);
    }
    /**
     * @param mixed $type 
     * @param string $path 
     * @param array $parameters 
     * @return ApiListResponse|false 
     * @throws Exception 
     */
    public static function requestList($type,$path="",$parameters=[],$className=null) {
        static::verifyInit();
        $url = static::getEndpoint($path);
        self::printHttpDebug("GET","Getting list from: ".$url);
        $options = [];
        $options["headers"] = static::guzzleHeaders();
        if(!empty($parameters) && is_array($parameters)) {
            $options['query'] = $parameters;
        }
        if($className === null) {
            $className = ListItem::class;
        }
        $client = new \GuzzleHttp\Client();
        $i=0; $isError = false;
        while($i++<3) {
            try {
                $response = $client->request("GET",$url,$options);
                $parsed = static::responseParse($response);
                self::printHttpDebug("GET","Completed");
                return ApiListResponse::fromListResponse($parsed,$className,[static::class,"requestList"],[$type,$path,$parameters,$className]);
            } catch(Exception $e) {
                $isError = $e;
                self::printHttpDebug("GET","Error with request.. retrying..");
            }
        }
        if($isError) {
            self::printHttpDebug("GET","Error with request.. throwing..");
            throw $isError;
        }
    }

    /**
     * @param string $path
     * @param array $paramaters
     * @return Request
     * @throws \Exception
     */
    public static function post($path="",$parameters=[]) {
        static::verifyInit();
        $url = static::getEndpoint($path);
        self::printHttpDebug("POST","Posting to: ".$url);
        $options = [];
        $options["headers"] = static::guzzleHeaders();
        if(!empty($parameters) && is_array($parameters)) {
            $options['json'] = $parameters;
        }
        $client = new \GuzzleHttp\Client();
        $i=0; $isError = false;
        while($i++<3) {
            try {
                $response = $client->request("POST",$url,$options);
                self::printHttpDebug("POST","Completed");
                return static::responseParse($response);
            } catch(Exception $e) {
                $isError = $e;
                self::printHttpDebug("POST","Error with request.. retrying..");
            }
        }
        if($isError) {
            self::printHttpDebug("POST","Error with request.. throwing..");
            throw $isError;
        }
    }
    /**
     * @param string $path
     * @param array $paramaters
     * @return ApiBaseResponse
     * @throws \Exception
     */
    public static function upload($path="",$parameters=[]) {
        static::verifyInit();
        $url = static::getEndpoint($path);
        self::printHttpDebug("UPLOAD","Uploading to: ".$url);
        $options = [];
        $options["headers"] = static::guzzleHeaders("multipart/form-data");
        foreach($parameters as $key=>$value) {
            $options["multipart"][] = [
                "name"=>$key,
                "contents"=>$value
            ];
        }
        $client = new \GuzzleHttp\Client();
        $response = $client->request("POST",$url,$options);
        $i=0; $isError = false;
        while($i++<3) {
            try {
                $body = $response->getBody()->getContents();
                self::printHttpDebug("UPLOAD","Completed");
                return ApiBaseResponse::fromArray($body);
            } catch(Exception $e) {
                $isError = $e;
                self::printHttpDebug("UPLOAD","Error with request.. retrying..");
            }
        }
        if($isError) {
            self::printHttpDebug("UPLOAD","Error with request.. throwing..");
            throw $isError;
        }
    }

    public static function getRelPath($from, $to, $ps = DIRECTORY_SEPARATOR)
    {
        $arFrom = explode($ps, rtrim($from, $ps));
        $arTo = explode($ps, rtrim($to, $ps));
        while(count($arFrom) && count($arTo) && ($arFrom[0] == $arTo[0])) {
            array_shift($arFrom);
            array_shift($arTo);
        }
        if(empty($arFrom) && count($arTo) === 1) {
            return ".".$ps.$arTo[0];
        }
        return str_pad("", count($arFrom) * 3, '..'.$ps).implode($ps, $arTo);
    }

    /**
     * @param string $path
     * @param array $paramaters
     * @return dhDB
     * @throws \Exception
     */
    public static function db($dbconfig=null) {
        if($dbconfig) {
            return dhGlobal::db($dbconfig);
        }
        if(!dhGlobal::db() && OpenAIConfig::get("dbconfig.use",false)) {
            return dhGlobal::db(OpenAIConfig::get("dbconfig"));
        }
        return dhGlobal::db();
    }

    public static function redis($redisconfig=null) {
        if($redisconfig) {
            $redis = new \Predis\Client($redisconfig);
            dhGlobal::set("redis",$redis);
            return $redis;
        }
        if(($redis= dhGlobal::get("redis",false)) !== false) {
            return $redis;
        }
        if(OpenAIConfig::get("redis")) {
            $config = OpenAIConfig::get("redis");
            if(is_array($config) && isset($config["passwrod"]) && !empty($config["password"])) {
                if(is_array($config["password"])) {
                    $arr = $config["password"];
                    unset($config["password"]);
                    $config["username"] = $arr[0];
                    $config["password"] = $arr[1];
                }
            }
            $redis = new \Predis\Client($config);
            dhGlobal::set("redis",$redis);
            return $redis;
        }
    }

    public static function config($key,$default=null) {
        return OpenAIConfig::get($key,$default);
    }
    public static function setConfig($key,$value) {
        return OpenAIConfig::set($key,$value);
    }
    public static function saveConfig() {
        return OpenAIConfig::save();
    }
    public static function table($tableName) {
        return OpenAIConfig::get("tables.$tableName",false);
    }

    public static function assertConfig() {
        if(OpenAIConfig::get("openai.api_key","") == "") {
            throw new \Exception("OpenAI API Key not set");
        }
    }

    public static function isCLI() {
        return php_sapi_name() == "cli";
    }

    public static function assistant($name) {
        if(is_object($name) && get_class($name) == Assistant::class) {
            return $name;
        }
        if(is_array($name) && isset($name["id"])) {
            return Assistant::fromArray($name);
        }
        
        $assistantId = OpenAIConfig::get("assistants.".$name,false);
        if($assistantId !== false) {
            return Assistant::fromId($assistantId);
        }

        if(substr($name,0,5) == "asst_") {
            return Assistant::fromId($name);
        }
        return Assistant::fromName($name);
        
    }

    public static function chat($options=[]) { 
        $chat = Chat::fromArray($options); 
        return $chat->send(); 
    }

    public static function printDebug(...$messages) {
        if(static::debugMode()) {
            array_unshift($messages,"[DEBUG]");
            Output::outLine(...$messages);
        }
    }
    public static function printHttpDebug(...$messages) {
        if(static::debugHttpMode()) {
            array_unshift($messages,"[HTTP]");
            Output::outLine(...$messages);
        }
    }
    public static function toUtf8($str) {
        return mb_convert_encoding($str, 'UTF-8', 'auto');
    }
    private static function responseParse($response) {
        $body = $response->getBody()->getContents();
        $json = json_decode($body,true);
        if($json === null) {
            return $body;
        }
        return $json;
    }
}