<?php
namespace boru\openai\embeddings\vectorstore;

use boru\output\Output;
use boru\openai\embeddings\distance\Cosine;
use boru\openai\embeddings\distance\Euclidean;
use boru\openai\embeddings\Document;
use boru\openai\models\Embedding;
use boru\openai\OpenAI;
use boru\openai\OpenAIConfig;

class FileVectorStore extends VectorStore {
    /** @var Document[] */
    private $documents = [];
    private $filePath;

    /** @var DistanceInterface */
    private $distance;

    public function __construct($filePath = null) {
        if($filePath !== null) {
            $this->filePath = $filePath;
        } else {
            $this->filePath = getcwd() . '/vectorstore_data.json';
        }
        $this->load();
    }

    public function distance($distance=null) {
        if($distance !== null) {
            $this->distance = $distance;
        }
        if($this->distance === null) {
            $this->distance = new Euclidean();
        }
        return $this->distance;
    }

    public function countDocuments() {
        return count($this->documents);
    }

    public function addDocument(Document $document) {
        //trigger embedding if not already embedded
        $document->embed();
        //add to the store
        $this->documents[$document->id()] = $document;
        //save the store
        $this->save();
    }

    public function addDocuments(array $documents) {
        foreach($documents as $document) {
            $this->addDocument($document);
        }
    }

    public function similarity($embedding, $k = 4, $extraArgs = []) {
        if(!is_array($embedding)) {
            $emb = new Embedding($embedding);
            $embedding = $emb->embedding();
        }
        $distances  = [];
        foreach($this->documents as $i=>$document) {
            $dist = $this->distance()->measure($embedding, $document->embedding());
            if($dist !== false) {
                $distances[$i] = $dist;
            }
        }
        asort($distances);

        $distances = array_slice(array_keys($distances), 0, $k, true);


        $results = [];
        foreach($distances as $i) {
            $results[] = $this->documents[$i];
        }
        return $results;
    }

    private function load() {
        if (file_exists($this->filePath)) {
            $data = json_decode(file_get_contents($this->filePath), true);
            if(is_array($data) && !empty($data)) {
                foreach($data as $documentData) {
                    $document = new Document($documentData);
                    $this->documents[$document->id()] = $document;
                }
            }
        }
    }

    private function save() {
        Output::outLine("saving with",count($this->documents),"documents");
        file_put_contents($this->filePath, json_encode($this->documents));
    }

    public function getDocuments() {
        return $this->documents;
    }
}