<?php
namespace boru\openai\models;

use boru\openai\api\endpoints\Assistants;
use boru\openai\api\endpoints\Threads;
use boru\openai\api\responses\AssistantResponse;
use boru\openai\OpenAI;
use boru\openai\OpenAIConfig;
use boru\openai\tiktoken\EncoderProvider;
use boru\output\Output;
use Exception;

class Assistant extends Base {
    /** @var string */
    private $id;
    /** @var string */
    private $object = "assistant";
    /** @var int */
    private $createdAt;
    /** @var string|null */
    private $name;
    /** @var string|null */
    private $description;
    /** @var string */
    private $model;
    /** @var string|null */
    private $instructions;
    /** @var array */
    private $tools;
    /** @var array */
    private $fileIds;
    /** @var array */
    private $metadata;
    /** @var string */
    private $tag;

    private $encoder;

    private $customPrompts = false;

    /** @var array */
    private $prompts = [];

    private static $tableName;
    public static function tableName($tableName=null) {
        if($tableName) {
            static::$tableName = $tableName;
        }
        if(static::$tableName === null) {
            static::$tableName = OpenAI::table("assistants","boru_openai_assistants");
        }
        return static::$tableName;
    }

    /** @var Thread */
    private $thread;

    public function id($id=null) {
        if ($id !== null && $id != "new") {
            $this->id = $id;
        }
        return $this->id;
    }
    public function object($object=null) {
        if ($object !== null) {
            $this->object = $object;
        } elseif($object === false) {
            $this->object = null;
        }
        return $this->object;
    }
    public function createdAt($created=null) {
        if ($created !== null) {
            $this->createdAt = $created;
        } elseif($created === false) {
            $this->createdAt = null;
        }
        return $this->createdAt;
    }
    public function name($name=null) {
        if ($name !== null) {
            $this->name = $name;
        } elseif($name === false) {
            $this->name = null;
        }
        return $this->name;
    }
    public function description($description=null) {
        if ($description !== null) {
            $this->description = $description;
        } elseif($description === false) {
            $this->description = null;
        }
        return $this->description;
    }
    public function model($model=null) {
        if ($model !== null) {
            $this->model = $model;
        } elseif($model === false) {
            $this->model = null;
        }
        return $this->model;
    }
    public function instructions($instructions=null) {
        if ($instructions !== null) {
            $this->instructions = $instructions;
        } elseif($instructions === false) {
            $this->instructions = null;
        }
        return $this->instructions;
    }
    public function tools($tools=null) {
        if ($tools !== null) {
            $this->tools = $tools;
        } elseif($tools === false) {
            $this->tools = null;
        }
        return $this->tools;
    }
    public function fileIds($fileIds=null) {
        if ($fileIds !== null) {
            $this->fileIds = $fileIds;
        } elseif($fileIds === false) {
            $this->fileIds = null;
        }
        return $this->fileIds;
    }
    public function metadata($metadata=null) {
        if ($metadata !== null) {
            $this->metadata = $metadata;
        } elseif($metadata === false) {
            $this->metadata = null;
        }
        return $this->metadata;
    }
    public function tag($tag=null) {
        if ($tag !== null) {
            $this->tag = $tag;
        } elseif($tag === false) {
            $this->tag = null;
        }
        return $this->tag;
    }
    public function encoder($encoder=null) {
        if ($encoder !== null) {
            $this->encoder = $encoder;
        } elseif($encoder === false) {
            $this->encoder = null;
        }
        if(!$this->encoder) {
            $provider = new EncoderProvider();
            $this->encoder = $provider->getForModel($this->model());
        }
        return $this->encoder;
    }
    public function encode($text) {
        return $this->encoder()->encode($text);
    }
    public function decode($text) {
        return $this->encoder()->decode($text);
    }
    public function isCustomPrompts($customPrompts=null) {
        if ($customPrompts !== null) {
            $this->customPrompts = $customPrompts ? true : false;
        }
        return $this->customPrompts;
    }
    /**
     * Return the prompt body as a string, or as an array of Prompt objects. Default is true (string).
     * @param $name string The name of the prompt to return
     * @param $promptBody bool If true, return the prompt body as a string. If false, return an array of Prompt objects. Default is true.
     * @return string|Prompt[]|false
     */
    public function getPrompt($name,$promptBody=true) {
        if(empty($this->prompts)) {
            $this->prompts();
        }
        $prompts = [];
        foreach($this->prompts as $prompt) {
            if($prompt->name() == $name && !empty($prompt->prompt())) {
                $prompts[] = $prompt;
            }
        }
        if($promptBody) {
            $content = "";
            foreach($prompts as $prompt) {
                $content .= $prompt->prompt()."\n";
            }
            return $content;
        }
        return !empty($prompts) ? $prompts : false;
    }
    public function setPrompt($name,$prompt) {
        if(empty($this->prompts)) {
            $this->prompts();
        }
        if(!is_object($prompt)) {
            $text = $prompt;
            $prompt = Prompt::forAssistant($this->id(),$name);
            if(!$prompt) {
                $prompt = new Prompt(["assistant_id"=>$this->id(),"name" => $name, "prompt" => $text]);
            }
            $prompt->prompt($text);
            $prompt->save();
        }
        $found = false;
        foreach($this->prompts as $key => $value) {
            if($value->name() == $name) {
                $this->prompts[$key] = $prompt;
                $found = true;
            }
        }
        if(!$found) {
            $this->prompts[] = $prompt;
        }
    }
    public function prompt($name,$prompt=null) {
        if($prompt) {
            $this->setPrompt($name,$prompt);
        }
        return $this->getPrompt($name);
    }
    public function prompts() {
        $prompts = Prompt::forAssistant($this->id);
        if(is_array($prompts) && !empty($prompts)) {
            $this->prompts = $prompts;
        }
    }
    public function addTool($tool) {
        //dedupe
        if(is_array($this->tools) && count($this->tools) > 0) {       
            foreach($this->tools as $key => $value) {
                if(isset($value["type"]) && $value["type"] == $tool) {
                    return $this;
                } elseif($value == $tool) {
                    return $this;
                }
            }
        }
        if(is_array($tool)) {
            $this->tools[] = $tool;
        } else {
            $this->tools[] = ["type" => $tool];
        }
        
    }

    public function save() {
        $parameters = [
            "model" => $this->model,
            "name" => $this->name,
            "description" => $this->description,
            "instructions" => $this->instructions,
            "tools" => $this->tools,
            "file_ids" => $this->fileIds,
            "metadata" => $this->metadata
        ];
        foreach($parameters as $key => $value) {
            if(!$value || $value == 'null' || $value == '[]' || (is_array($value) && count($value) == 0)) {
                unset($parameters[$key]);
            }
        }
        if($this->id) {
            $result = Assistants::update($this->id,$parameters);
            if($result->id()) {
                $this->id = $result->id();
                $this->saveToTable();
                return $this;
            }
        } else {
            $result = Assistants::create($parameters);
            $this->id = $result->id();
            $this->saveToTable();
            return $this;
        }
        return false;
        //return $this->request("patch","assistants/".$this->id,$parameters);
    }

    public function saveToTable() {
        $db = OpenAI::db();
        if($db === false) {
            return;
        }
        //$db->printDebug(true);
        $db->query("INSERT INTO `".static::tableName()."` (`id`,`object`,`tag`,`model`,`name`,`description`,`instructions`,`tools`,`file_ids`,`metadata`) VALUES (?,?,?,?,?,?,?,?,?,?) ON DUPLICATE KEY UPDATE `tag`=?,`model`=?,`name`=?,`description`=?,`instructions`=?,`tools`=?,`file_ids`=?,`metadata`=?",
            [
                $this->id,
                $this->object,
                $this->tag,
                $this->model,
                $this->name,
                $this->description,
                $this->instructions,
                json_encode($this->tools),
                json_encode($this->fileIds),
                json_encode($this->metadata),
                $this->tag,
                $this->model,
                $this->name,
                $this->description,
                $this->instructions,
                json_encode($this->tools),
                json_encode($this->fileIds),
                json_encode($this->metadata)
            ]
        );
        if(!empty($this->tools) && is_array($this->tools)) {
            foreach($this->tools as $tool) {
                if(isset($tool["type"]) && $tool["type"] == "function") {
                    $checkTool = ToolFunction::fromName($tool["function"]["name"],$this->id);
                    if(!$checkTool) {
                        $toolFunction = new ToolFunction();
                        $toolFunction->setFromData($tool["function"]);
                        $toolFunction->assistantId($this->id);
                        $toolFunction->save();
                    } else {
                        $checkTool->setFromData($tool["function"]);
                        $checkTool->save();
                    }
                }
            }
        }
    }
    public function delete() {
        $result = Assistants::delete($this->id);
        if($result) {
            $db = OpenAI::db();
            if($db === false) {
                return;
            }
            $db->query("DELETE FROM `".static::tableName()."` WHERE `id`=?",[$this->id]);
        }
        return $result;
    }

    public function reload($useDb=false) {
        $copy = static::fromId($this->id,$useDb);
        $this->setFromData($copy->toArray());
        return $this;
    }

    /**
     * 
     * @return AssistantResponse|Run|string|false 
     * @param $concatenate string - the string to concatenate messages with, false to return an iteratable object AssestantResponse
     * @param $returnRun bool - if true, return the Run object instead of the result
     * @throws Exception 
     */
    public function run($concatenate=false,$returnRun=false,$keepThread=false) {
        if(!$this->thread) {
            throw new \Exception("Thread not found. Add a message first to create one.");
        }

        $beforeMessage = false;
        $afterMessage = false;
        if(!$this->isCustomPrompts()) {
            $this->prompts();
            if($this->prompt("before") && !empty($this->getPrompt("before"))) {
                $data = [
                    "role" => "user",
                    "content" => $this->getPrompt("before"),
                    "attachments" => [],
                    "metadata" => []
                ];
                $beforeMessage = new Message($data);
            }
            if($this->prompt("after") && !empty($this->getPrompt("after"))) {
                $data = [
                    "role" => "user",
                    "content" => $this->getPrompt("after"),
                    "attachments" => [],
                    "metadata" => []
                ];
                $afterMessage = new Message($data);
            }
        }
        if($beforeMessage || $afterMessage) {
            $messageArray = $this->thread->getMessages(false);
            if($beforeMessage) {
                array_unshift($messageArray,$beforeMessage);
            }
            if($afterMessage) {
                $messageArray[] = $afterMessage;
            }
            foreach($messageArray as $message) {
                $messages[] = $message->forThread();
            }
        } else {
            $messages = $this->thread->getMessages();
        }
        

        $parameters = [
            "assistant_id" => $this->id,
            "thread" => [
                "messages" => $messages
            ]
        ];
        //print_r($parameters);
        //exit();
        $run = Threads::run($parameters);
        
        if($returnRun) {
            if(!$keepThread) {
                $this->thread = null;
            }
            return $run;
        }
        
        $response = new AssistantResponse($run->result());
        if(!$keepThread) {
            $this->thread = null;
        }
        if($concatenate === false) {
            return $response;
        }
        
        $output = "";
        foreach($response as $message) {
            $output .= $message->value().$concatenate;
        }
        return $output;
    }

    public function output($concatenate=false) {
        return $this->run($concatenate);
    }
    public function json($asArray=false) {
        $output = trim($this->output("\n"));
        if(substr($output,0,7)=="```json") {
            $output = ltrim(substr($output,7));
        }
        if(substr($output,-3)=="```") {
            $output = rtrim(substr($output,0,-3));
        }
        if($asArray) {
            return json_decode($output, true);
        }
        return $output;
    }
    public function addMessage($role,$content,$attachments=[],$metadata=[]) {
        if(!$this->thread) {
            $this->thread = new Thread();
        }
        $this->thread->addMessage($role,$content,$attachments,$metadata);
    }

    public static function queryByName($name) {
        $db = OpenAI::db();
        if($db === false) {
            return false;
        }
        $result = $db->query("SELECT * FROM `".static::tableName()."` WHERE `name`=? LIMIT 1",[$name]);
        while($row = $db->next($result)) {
            $array = static::cleanTableRowForAssistat($row->asArray());
            return new Assistant($array);
        }
        return false;
    }
    public static function queryById($id) {
        $db = OpenAI::db();
        if($db === false) {
            return false;
        }
        $result = $db->query("SELECT * FROM `".static::tableName()."` WHERE `id`=? LIMIT 1",[$id]);
        while($row = $db->next($result)) {
            $array = static::cleanTableRowForAssistat($row->asArray());
            return new Assistant($array);
        }
        return false;
    }
    public static function queryByTag($tag) {
        $db = OpenAI::db();
        if($db === false) {
            return false;
        }
        $result = $db->query("SELECT * FROM `".static::tableName()."` WHERE `tag`=? LIMIT 1",[$tag]);
        while($row = $db->next($result)) {
            $array = static::cleanTableRowForAssistat($row->asArray());
            return new Assistant($array);
        }
        return false;
    }
    public static function fromTable($idOrNameOrTag,$which="all") {
        $db = OpenAI::db();
        if($db === false) {
            return Assistants::getAssistant($idOrNameOrTag);
        }
        if($which === "all") {
            $assistant = static::queryById($idOrNameOrTag);
            if(!$assistant) {
                $assistant = static::queryByName($idOrNameOrTag);
            }
            if(!$assistant) {
                $assistant = static::queryByTag($idOrNameOrTag);
            }
            return $assistant;
        }
        if($which == "id") {
            return static::queryById($idOrNameOrTag);
        }
        if($which == "name") {
            return static::queryByName($idOrNameOrTag);
        }
        if($which == "tag") {
            return static::queryByTag($idOrNameOrTag);
        }
        return false;
    }
    private static function cleanTableRowForAssistat($rowArray) {
        $rowArray["tools"] = json_decode($rowArray["tools"],true);
        $rowArray["file_ids"] = json_decode($rowArray["file_ids"],true);
        $rowArray["metadata"] = json_decode($rowArray["metadata"],true);
        return $rowArray;
    }

    /**
     * 
     * @param $name string
     * @param $useTableCache bool
     * @return Assistant|false
     */
    public static function fromName($name,$useTableCache=true) {
        $assistant = false;
        if($useTableCache) {
            $assistant = static::fromTable($name,"name");
            if($assistant) {
                return $assistant;
            }
        }
        $assistant = Assistants::byName($name);
        if($assistant) {
            if($useTableCache) {
                $assistant->saveToTable();
            }
            return $assistant;
        }
        if(OpenAI::config("assistants.$name",false)) {
            $assistant = static::fromId(OpenAI::config("assistants.$name"));
            if($assistant) {
                if($useTableCache) {
                    $assistant->saveToTable();
                }
                return $assistant;
            }
        }
        if(file_exists(__DIR__."/../../assistants/".$name.".json")) {
            echo "Having to create assistant from file: ".$name."\n";
            $assistant = static::fromFile(__DIR__."/../../assistants/".$name.".json");
            if($assistant) {
                if($useTableCache) {
                    $assistant->tag($name);
                    $assistant->saveToTable();
                }
                return $assistant;
            }
        }
        return false;
    }

    public static function fromOpenAI($assistantNameOrId) {
        try {
            $assistant = Assistants::getAssistant($assistantNameOrId);
        } catch(Exception $e) {
            $assistant = false;
        }
        if($assistant) {
            $assistant->saveToTable();
            return $assistant;
        }
        $assistant = Assistants::byName($assistantNameOrId);
        if($assistant) {
            $assistant->saveToTable();
            return $assistant;
        }
        return $assistant;
    }

    /**
     * 
     * @param $id string
     * @param $useTableCache bool
     * @return Assistant|false
     */
    public static function fromId($id,$useTableCache=true) {
        $assistant = false;
        if(strlen($id) >= 50) {
            return false;
        }
        if($useTableCache) {
            $assistant = static::fromTable($id,"id");
        }
        if(!$assistant) {
            $assistant = Assistants::getAssistant($id);
            if($assistant && $useTableCache) {
                $assistant->saveToTable();
            }
        }
        return $assistant ? $assistant : false;
    }

    /**
     * 
     * @param $input string|Assistant|array
     * @return Assistant|false
     */
    public static function fromInput($input,$useDb=true) {
        if($input instanceof Assistant) {
            return $input;
        }
        if(is_array($input)) {
            return new Assistant($input);
        }
        if(is_string($input)) {
            $json = json_decode($input,true);
            if($json) {
                return new Assistant($json);
            }
            $assistant = false;
            if($useDb) {
                $assistant = Assistant::fromTable($input);
                if($assistant) {
                    return $assistant;
                }
            }
            
            if(file_exists($input)) {
                $assistant = Assistant::fromFile($input);
                if($assistant) {
                    return $assistant;
                }
            }
            
            try {
                $assistant = Assistant::fromId($input);
            } catch(Exception $e) {
                $assistant = false;
            }

            if($assistant) {
                return $assistant;
            }
            echo "Having to fall back to fromName::".$input."\n";
            return Assistant::fromName($input);
        }
    }

    public function toArray() {
        return [
            "id" => $this->id,
            "object" => $this->object,
            "created_at" => $this->createdAt,
            "name" => $this->name,
            "description" => $this->description,
            "model" => $this->model,
            "instructions" => $this->instructions,
            "tools" => $this->tools,
            "file_ids" => $this->fileIds,
            "metadata" => $this->metadata
        ];
    }

    public function toFile($fileName) {
        file_put_contents($fileName,json_encode($this->toArray(),JSON_PRETTY_PRINT));
    }
    public function readFile($fileName) {
        if(file_exists($fileName)) {
            $data = json_decode(file_get_contents($fileName),true);
            foreach($data as $key => $value) {
                $this->$key = $value;
            }
        }
    }
    public static function fromFile($fileName,$forceCreate=false) {
        $data = json_decode(file_get_contents($fileName),true);
        $assistant = self::fromArray($data,$forceCreate);
        return $assistant;
    }
    public static function fromArray($data,$forceCreate=false) {
        if(is_object($data)) {
            $data = $data->toArray();
        }
        if(!is_array($data)) {
            $data = json_decode($data,true);
        }
        if(!is_array($data)) {
            throw new \Exception("Input for Assistant::fromArray must be an array, json_string, or an object that can be converted to an array.");
        }
        foreach($data as $key => $value) {
            if($value === null) {
                unset($data[$key]);
            }
        }
        if(!$forceCreate && isset($data["id"])) {
            return new Assistant($data);
        }
        if(isset($data["id"])) {
            unset($data["id"]);
        }
        if(isset($data["created_at"])) {
            unset($data["created_at"]);
        }
        $assistant = new Assistant($data);
        $assistant->save();
        return $assistant;
    }

    public static function migrateFromConfig() {
        $assistants = OpenAIConfig::get("assistants",[]);
        if(count($assistants) > 0) {
            Output::outLine("Migrating assistants from config to database");
            foreach($assistants as $tag=>$assistantId) {
                if(empty($assistantId)) {
                    OpenAIConfig::delete("assistants.".$tag);
                    continue;
                }
                $check = Assistant::fromTable($tag,"tag");
                if(!$check) {
                    $check = Assistant::fromTable($assistantId,"id");
                    if($check) {
                        $check->tag($tag);
                        $check->saveToTable();
                    }
                }
                if(!$check) {
                    $assistant = Assistant::fromOpenAI($assistantId);
                    if($assistant) {
                        $assistant->saveToTable();
                        if(OpenAI::isCLI()) {
                            Output::outLine("Removing assistant $assistantId from config");
                        }
                        OpenAIConfig::delete("assistants.".$tag);
                    }
                } else {
                    if(OpenAI::isCLI()) {
                        Output::outLine("Removing assistant $assistantId from config");
                    }
                    OpenAIConfig::delete("assistants.".$tag);
                }
            }
            OpenAIConfig::save();
        }
        if(empty(OpenAIConfig::get("assistants"))) {
            OpenAIConfig::delete("assistants");
            OpenAIConfig::save();
        }
    }
}