<?php
namespace boru\openai\models\document;

use boru\openai\models\Assistant;
use boru\openai\models\Base;
use boru\openai\OpenAI;
use boru\openai\tools\OCR;
use boru\output\Output;

class Page extends Base {
    private static $tableName;
    private $id;
    private $docid;
    private $openai_fileid = "";
    private $created;
    private $uploaded;
    private $processed;
    private $content;
    private $contentTokens = 0;

    private $ocrDetail = "high";

    public static function tableName($tableName=null) {
        if($tableName) {
            static::$tableName = $tableName;
        }
        if(static::$tableName === null) {
            static::$tableName = OpenAI::table("pages","boru_openai_pages");
        }
        return static::$tableName;
    }

    public function __construct($idOrArray=null) {
        if ($idOrArray !== null) {
            if (is_array($idOrArray)) {
                $this->loadFromArray($idOrArray);
            } else if (is_numeric($idOrArray)) {
                $this->loadFromId($idOrArray);
            }
        }
    }

    public function id() {
        return $this->id;
    }
    public function docid($docid=null) {
        if ($docid !== null) {
            $this->docid = $docid;
            return $this;
        }
        return $this->docid;
    }
    public function openai_fileid($openai_fileid=null) {
        if ($openai_fileid !== null) {
            $this->openai_fileid = $openai_fileid;
            return $this;
        }
        return $this->openai_fileid;
    }
    public function created($created=null) {
        if ($created !== null) {
            $this->created = $created;
            return $this;
        }
        return $this->created;
    }
    public function uploaded($uploaded=null) {
        if ($uploaded !== null) {
            $this->uploaded = $uploaded;
            return $this;
        }
        return $this->uploaded;
    }
    public function processed($processed=null) {
        if ($processed !== null) {
            $this->processed = $processed;
            return $this;
        }
        return $this->processed;
    }
    public function content($content=null) {
        if ($content !== null) {
            $this->content = $content;
            return $this;
        }
        return $this->content;
    }
    public function contentTokens($tokens=null) {
        if ($tokens !== null) {
            $this->contentTokens = $tokens;
            return $this;
        }
        return $this->contentTokens;
    }
    public function isProcessed() {
        return $this->processed !== null;
    }
    public function ocrDetail($detail=null) {
        if ($detail !== null) {
            $this->ocrDetail = $detail == "high" ? "high" : "low";
            return $this;
        }
        return $this->ocrDetail == "high" ? "high" : "low";
    }
    public function highDetail() {
        return $this->ocrDetail("high");
    }
    public function lowDetail() {
        return $this->ocrDetail("low");
    }

    /**
     * @param bool $force
     * @param Assistant|string $assistant
     * @return string
     */
    public function ocr($force=false,$assistant=null) {
        if(!$force && $this->isProcessed()) {
            return $this->content();
        }
        if($assistant) {
            $assistant = Assistant::fromInput($assistant);
        }
        if(!$assistant) {
            $assistant = Assistant::fromInput("ocr");
        }
        $ocr = new OCR(["detail"=>$this->ocrDetail()]);
        $ocr->fileId($this->openai_fileid());
        $result = $ocr->run();
        $this->content($result);
        $this->processed(date("Y-m-d H:i:s"));
        if($this->content()) {
            $this->contentTokens($ocr->tokenCount());
        }
        $this->save();
        return $result;
    }

    public function ocrTokens($force=false,$assistant=null) {
        if(!$force && $this->contentTokens() > 0) {
            return $this->contentTokens();
        }
        if($assistant) {
            $assistant = Assistant::fromInput($assistant);
        }
        if(!$assistant) {
            $assistant = Assistant::fromInput("ocr");
        }
        if(empty($this->content())) {
            Output::outLine("No content to tokenize");
            return 0;
            //$this->ocr();
        }
        if(empty($this->content())) {
            return 0;
        }
        $tokens = $assistant->encode($this->content());
        $this->contentTokens(count($tokens));
        $this->save();
    }

    public function save() {
        $db = OpenAI::db();
        if ($this->id) {
            $db->query("UPDATE ".static::tableName()." SET docid=?, openai_fileid=?, created=?, uploaded=?, processed=?, `content`=?, `content_tokens`=? WHERE id=?", [
                $this->docid, $this->openai_fileid, $this->created, $this->uploaded, $this->processed, $this->content, $this->contentTokens, $this->id
            ]);
        } else {
            $db->query("INSERT INTO ".static::tableName()." (docid, openai_fileid, created, uploaded, processed, `content`, `content_tokens`) VALUES (?, ?, NOW(), NOW(), ?, ?, ?)", [
                $this->docid, $this->openai_fileid, $this->processed, $this->content, $this->contentTokens
            ]);
            $this->id = $db->lastInsertId();
            $this->loadFromId($this->id);
        }
        return $this;
    }

    public function loadFromArray($array) {
        $this->id = $array['id'];
        $this->docid = $array['docid'];
        $this->openai_fileid = $array['openai_fileid'];
        $this->created = $array['created'];
        $this->uploaded = $array['uploaded'];
        $this->processed = $array['processed'];
        $this->content = $array['content'];
        $this->contentTokens = $array['content_tokens'];
    }
    public function loadFromId($id) {
        $page = static::queryById($id);
        $this->loadFromArray($page);
    }
    public function toArray() {
        return [
            'id' => $this->id,
            'docid' => $this->docid,
            'openai_fileid' => $this->openai_fileid,
            'created' => $this->created,
            'uploaded' => $this->uploaded,
            'processed' => $this->processed,
            'content' => $this->content,
        ];
    }

    public static function getNeedsOCR() {
        $db = OpenAI::db();
        $res = $db->run("SELECT * FROM ".static::tableName()." WHERE uploaded IS NOT NULL AND processed IS NULL ORDER BY `created` ASC", []);
        $docs = [];
        while($row = $db->next($res)) {
            $docs[] = new static($row->asArray());
        }
        return $docs;
    }
    public static function getNeedsOCRTokens() {
        $db = OpenAI::db();
        $res = $db->run("SELECT * FROM ".static::tableName()." WHERE uploaded IS NOT NULL AND processed IS NOT NULL AND content_tokens=0 ORDER BY `processed` DESC", []);
        $docs = [];
        while($row = $db->next($res)) {
            $docs[] = new static($row->asArray());
        }
        return $docs;
    }

    public static function queryById($id) {
        $db = OpenAI::db();
        $res = $db->run("SELECT * FROM ".static::tableName()." WHERE id=?", [$id]);
        while($row = $db->next($res)) {
            return $row->asArray();
        }
        return false;
    }
    public static function queryByDocId($docId) {
        $db = OpenAI::db();
        $res = $db->query("SELECT * FROM ".static::tableName()." WHERE docid=? ORDER BY `id` ASC", [$docId]);
        $pages = [];
        while($row = $db->next($res)) {
            $page = new Page();
            $page->loadFromArray($row->asArray());
            $pages[] = $page;
        }
        return $pages;
    }

    public static function create($docId,$fileId) {
        $page = new Page();
        $page->docid($docId);
        $page->openai_fileid($fileId);
        $page->save();
        return $page;
    }
}