<?php
namespace boru\openai\tools;

use boru\openai\api\endpoints\Assistants;
use boru\openai\api\endpoints\Files;
use boru\openai\models\Assistant;
use boru\openai\models\document\Page;
use boru\openai\OpenAI;
use Exception;

class Summarize {

    private $defaultPrompts = [
        "prefix" => "This is part of a larger document that I need to summarize. There are {TOTAL_PAGES} pages in the document. Do not provide a page by page summary, but rather a summary of the entire document.",
        "context" => "Here is the summary of the previous {DONE_PAGES} pages:\n{SUMMARY}",
        "update" => "Using the previous summary, please create an updated summary adding the attached page(s). Retain the previous summary and add/update the new information. Keep the summary brief enough to utilize in further prompts, while retaining important information.",
        "new" => "Please write a summary of the attached page(s). Keep the summary brief enough to utilize in further prompts, while retaining important information.",
    ];

    private $fileIds = [];
    private $pageIds = [];
    private $textSections = [];
    private $detail = "high";
    private $assistantId;
    private $assistantName;

    /**
     * @var Assistant
     */
    private $assistant;

    private $context = "";

    private $pagesPerRun = 7;

    public $debugSteps = false;
    
    /**
     * 
     * @param array $options Valid options are fileIds, assistantId, assistantName, prompts
     * @return void 
     * @throws Exception 
     */
    public function __construct($options=[]) {
        if(!is_array($options)) {
            throw new \Exception("Options must be an array");
        }
        $this->fileIds = isset($options["fileIds"]) ? $options["fileIds"] : [];
        $this->pageIds = isset($options["pageIds"]) ? $options["pageIds"] : [];
        $this->textSections = isset($options["texts"]) ? $options["texts"] : [];
        $assistantName = isset($options["assistantName"]) ? $options["assistantName"] : null;
        if($assistantName) {
            $this->assistantName($assistantName);
        }
        $assistantId = isset($options["assistantId"]) ? $options["assistantId"] : null;
        if($assistantId) {
            $this->assistantId($assistantId);
        }
        if(!$this->assistant) {
            $assistantId = Assistant::queryByTag("summary");
            if($assistantId !== false) {
                $this->assistantId($assistantId);
            }
        }
        $this->detail = isset($options["detail"]) ? $options["detail"] : "high";
        $prompts = isset($options["prompts"]) ? $options["prompts"] : null;
        if($prompts) {
            $this->prompts($prompts);
        }
    }
    public function fileIds($fileIds=null) {
        if ($fileIds) {
            $this->fileIds = $fileIds;
        }
        return $this->fileIds;
    }
    public function pageIds($pageIds=null) {
        if ($pageIds) {
            $this->pageIds = $pageIds;
        }
        return $this->pageIds;
    }
    public function texts($text=null) {
        if ($text) {
            if(!is_array($text)) {
                throw new \Exception("Text must be an array of strings");
            }
            $this->textSections = $text;
        }
        return $this->textSections;
    }
    public function assistantName($assistantName=null) {
        if ($assistantName) {
            $assistant = Assistant::fromName($assistantName);
            if(!$assistant) {
                throw new \Exception("Assistant not found");
            }
            $this->assistantName = $assistant->name();
            $this->assistantId = $assistant->id();
            $this->assistant = $assistant;
        }
        return $this->assistantName;
    }
    public function assistantId($assistantId=null) {
        if ($assistantId) {
            if(is_object($assistantId)) {
                $this->assistant = $assistantId;
                $this->assistantId = $assistantId->id();
            } else {
                $assistant = Assistant::fromId($assistantId);
                if(!$assistant) {
                    throw new \Exception("Assistant not found");
                }
                $this->assistantName = $assistant->name();
                $this->assistantId = $assistant->id();
                $this->assistant = $assistant;
            }
        }
        return $this->assistantId;
    }
    public function detail($detail=null) {
        if ($detail) {
            $this->detail = $detail == "low" ? "low" : "high";
        }
        return $this->detail == "low" ? "low" : "high";
    }
    public function highDetail() {
        return $this->detail("high");
    }
    public function lowDetail() {
        return $this->detail("low");
    }
    
    public function prompt($type,$data=null) {
        if(!isset($this->defaultPrompts[$type])) {
            throw new \Exception("Prompt type not found");
        }
        if($data) {
            $this->assistant->setPrompt($type,$data);
        }
        $prompt = $this->assistant->getPrompt($type);
        if(!$prompt) {
            $this->assistant->setPrompt($type,$this->defaultPrompts[$type]);
            $prompt = $this->assistant->getPrompt($type);
        }
        return $prompt;
    }
    public function prompts($prompts=null) {
        if($prompts) {
            if(!is_array($prompts)) {
                throw new \Exception("Prompts must be an array");
            }
            foreach($prompts as $key => $value) {
                $this->prompt($key,$value);
            }
        }
        return $this->assistant->prompts();
    }

    public function run($force=false) {
        if(count($this->pageIds) > 0) {
            return $this->runIds("page",$force);
        }
        if(count($this->fileIds) > 0) {
            return $this->runIds("file",$force);
        }
        if(count($this->textSections) > 0) {
            return $this->runIds("text",$force);
        }
        throw new \Exception("No pages or files to summarize");
    }
    
    private function runIDs($type="file",$force=false) {
        if($force && !empty($this->context)) {
            $this->context = "";
        }
        if(!empty($this->context)) {
            return $this->context;
        }
        if($type == "file") {
            if(count($this->fileIds) == 0) {
                throw new \Exception("No files to summarize");
            }
            $ids = $this->fileIds;
        } elseif($type == "page") {
            if(count($this->pageIds) == 0) {
                throw new \Exception("No pages to summarize");
            }
            $ids = $this->pageIds;
        } elseif($type == "text") {
            if(count($this->textSections) == 0) {
                throw new \Exception("No text to summarize");
            }
            $ids = $this->textSections;
        }else {
            throw new \Exception("Invalid type");
        }
        
        if(!$this->assistant) {
            $this->assistant = Assistant::fromName("summary");
            if(!$this->assistant) {
                throw new \Exception("No assistant set");
            }
        }
        $this->assistant->isCustomPrompts(true);
        $this->context = "";

        $chunks = $this->makeChunks($type,$ids);
        if($chunks === false) {
            $this->doSummarize($type,$ids);
            return $this->context;
        }
        if(count($chunks)===1) {
            $this->context = $chunks[0];
        } else {
            $this->doSummarize("text",$chunks);
        }
        return $this->context;
    }

    private function doSummarize($type,$ids) {
        $donePages = 0;
        $doneIds = [];
        $totalPages = count($ids);
        foreach($ids as $i => $id) {
            $doneIds[] = $id;
            $donePages++;
            if($donePages % $this->pagesPerRun == 0 || $donePages == count($ids)) {
                $this->summarizeCall($type,$doneIds,$totalPages,$donePages);
                if($this->debugSteps) {
                    echo "Done Pages: $donePages\n";
                    echo "$this->context\n";
                    echo "-----------------------------------\n";
                }
                $doneIds = [];
            }
        }
        return $this->context;
    }

    private function makeChunks($type,$ids,$tokenLimit=38000) {
        if($type == "file") {
            return false;
        }
        $chunks = [];
        $content = "";
        $tokens = 0;
        $i=0;
        foreach($ids as $item) {
            $thisContent = "";
            if($type == "page") {
                $page = new Page($item);
                $thisContent = $page->ocr();
                $tokens += $page->contentTokens();
                $content .= "Page $i:\n".$thisContent."\n\n";
            } elseif($type == "text") {
                $thisContent = $item;
                $tokens += count($this->assistant->encode($thisContent));
                $content .= "Text $i:\n".$thisContent."\n\n";
            }
            $i++;
            if($tokens > $tokenLimit) {
                $chunks[] = $content;
                $content = "";
                $tokens = 0;
            }   
        }
        if($content) {
            $chunks[] = $content;
        }
        return $chunks;
    }

    private function summarizeCall($type="file",$ids,$totalPages,$donePages) {
        $prompt = str_replace("{TOTAL_PAGES}",$totalPages,$this->prompt("prefix"));
        $prompt = str_replace("{THIS_PAGES}",count($ids),$prompt);
        $prefix = [
            "type"=> "text",
            "text"=>$prompt,
        ];
        $content = [];
        $content[] = $prefix;
        if(!empty($this->context)) {
            $content[] = [
                "type"=>"text",
                "text"=>str_replace("{SUMMARY}",$this->context,str_replace("{DONE_PAGES}",$donePages,$this->prompt("context"))),
            ];
            $content[] = [
                "type"=>"text",
                "text"=>$this->prompt("update"),
            ];
        } else {
            $content[] = [
                "type"=>"text",
                "text"=>$this->prompt("new"),
            ];
        }
        foreach($ids as $id) {
            if($type == "file") {
                $content[] = [
                    "type"=>"image_file",
                    "image_file"=>["file_id"=>$id,"detail"=>$this->detail()],
                ];
            } elseif($type == "page") {
                $page = new Page($id);
                $content[] = [
                    "type"=>"text",
                    "text"=>"PAGE CONTENT:\n\n".$page->ocr(),
                ];
            } elseif($type == "text") {
                $content[] = [
                    "type"=>"text",
                    "text"=>$id,
                ];
            }
        }
        $assistant = clone $this->assistant;
        $assistant->addMessage("user",$content);
        $result = $assistant->run();
        //print_r($result);
        $this->context = "";
        foreach($result as $message) {
            $this->context .= $message->value()."\n";
        }

        //echo "Current Context: $currentContext\n\n";
        return $this->context;
    }
}