<?php
namespace boru\openai\cli\defs;

use boru\cli\CLI;
use boru\cli\params\Flag;
use boru\cli\params\Option;
use boru\cli\params\Positional;
use boru\output\Output;
use boru\openai\api\endpoints\Assistants;
use boru\openai\models\Assistant;

trait AssistantCLI {
    public function def_assistant() {
        return [
            "description"=> "Assistant Operations",
            "params" => [
                //Positional::create("command|The command to execute",["options"=>["list","get","create","update","delete"]]),
            ]
        ];
    }

    public function def_assistant_list() {
        return [
            "description"=> "List Assistants",
            "params" => [
            ]
        ];
    }
    public function cmd_assistant_list($cmd) {
        $assistants = Assistants::listAssistants();
        foreach($assistants as $assistant) {
            Output::outLine($assistant->id(),"\t",$assistant->name());
        }
    }

    public function def_assistant_get() {
        return [
            "description"=> "Get Assistant",
            "params" => [
                Positional::create("assistant|Assistant ID or Name"),
                Flag::create("db|Get the assistant from the database"),
                Option::create("f|file|The file to write the assistant to"),
            ]
        ];
    }
    public function cmd_assistant_get($cmd) {
        $assistantInput = $this->get("assistant");
        if($this->get("db",false)) {
            $assistant = Assistant::fromTable($assistantInput);
        } else {
            $assistant = Assistant::fromOpenAI($assistantInput);
        }
        if($assistant) {
            $assistant->saveToTable();
            $outFile = $this->get("file",false);
            if($outFile) {
                $assistant->toFile($outFile);
            }
            $data=json_encode($assistant->toArray(),JSON_PRETTY_PRINT);
            Output::outLine($data);
        } else {
            Output::outLine("Assistant",$assistantInput,"not found");
        }
    }

    public function def_assistant_create() {
        return [
            "description"=> "Create Assistant",
            "params" => [
                Flag::create("i|noninteractive|Run in non-interactive mode"),
                Option::create("f|file|The file to read from.. <assistant> must be the ID of the assistant to update"),
                Option::create("n|name|The name of the assistant"),
                Option::create("m|model|The model to use for the assistant",["default"=>"gpt-4o"]),
                Flag::create("filesearch|Enable the file search tool"),
                Flag::create("codeinterpreter|Enable the code interpreter tool"),
            ]
        ];
    }
    public function cmd_assistant_create($cmd) {
        $nonInteractive = $this->get("noninteractive",false);
        if(($assistantFile = $this->get("file",false))) {
            if(file_exists($assistantFile)) {
                $assistant = Assistant::fromFile($assistantFile);
            } else {
                throw new \Exception("Assistant JSON file is required to update assistant when using the -f|--file option");
            }
        } else {
            $assistant = new Assistant();
        }
        if(($name = $this->get("name",false))!==false) {
            $assistant->name($name);
        } elseif($nonInteractive) {
            $name = $assistant->name();
            if(!$name) {
                throw new \Exception("Name is required in non-interactive mode");
            }
        } else {
            $name = $assistant->name();
            if($name) {
                Output::outLine("Name:",$name);
                $newName = CLI::promptInput("Enter a new name, or press return to keep: ");
                if($newName) {
                    $name = $newName;
                }
            } else {
                $name = CLI::promptInput("Name: ");
            }
        }

        if(($model = $this->get("model",false))!==false) {
            $assistant->model($model);
        } elseif($nonInteractive) {
            $model = $assistant->model();
            if(!$model) {
                throw new \Exception("Model is required in non-interactive mode");
            }
        } else {
            $model = $assistant->model();
            if($model) {
                Output::outLine("Model:",$model);
                $newModel = CLI::promptInput("Enter a new model, or press return to keep: ");
                if($newModel) {
                    $model = $newModel;
                }
            } else {
                $model = CLI::promptInput("Model: ");
            }
        }

        if(!$nonInteractive) {
            $instructions = trim($assistant->instructions());
            $assistant->instructions(trim(CLI::edit($instructions)));
        }

        if($this->get("filesearch",false)) {
            $assistant->addTool("file-search");
        }
        if($this->get("codeinterpreter",false)) {
            $assistant->addTool("code-interpreter");
        }
        $assistant->save();
        $assistant->saveToTable();
    }

    public function def_assistant_update() {
        return [
            "description"=> "Update Assistant",
            "params" => [
                Positional::create("assistant|Assistant ID or Name"),
                Option::create("f|file|The file to read from.. <assistant> must be the ID of the assistant to update"),
            ]
        ];
    }
    public function cmd_assistant_update($cmd) {
        $assistantInput = $this->get("assistant");
        if(($assistantFile = $this->get("file",false))) {
            if(file_exists($assistantFile)) {
                $assistant = Assistant::fromFile($assistantFile);
                $assistant->id($assistantInput);
            } else {
                throw new \Exception("Assistant JSON file is required to update assistant when using the -f|--file option");
            }
        } else {
            $assistant = Assistant::fromInput($assistantInput);
        }
        if($assistant) {
            $assistant->save();
            Output::outLine("Assistant",$assistant->id(),"updated");
        }
    }

    public function def_assistant_edit() {
        return [
            "description"=> "Update Assistant",
            "params" => [
                Positional::create("assistant*|Assistant ID or Name"),
                Positional::create("attribute*|The attribute to update",["options"=>["name","description","instructions"]]),
            ]
        ];
    }
    public function cmd_assistant_edit($cmd) {
        $assistantInput = $this->get("assistant");
        $assistant = Assistant::fromInput($assistantInput);
        if(!$assistant) {
            Output::outLine("Assistant",$assistantInput,"not found");
            return;
        }
        $attribute = $this->get("attribute");
        if(!in_array($attribute,["name","description","instructions"])) {
            Output::outLine("Attribute",$attribute,"not found");
            return;
        }
        
        $original = trim(call_user_func([$assistant,$attribute]));
        $result = trim(CLI::edit($original));
        if($result != $original) {
            call_user_func_array([$assistant,$attribute],[$result]);
            $assistant->save();
            $assistant->saveToTable();
            Output::outLine("Assistant",$assistant->id(),"updated");
            Output::outLine("Changed",$attribute);
            Output::outLine("From:",$original);
            Output::outLine("To:",$result);
        }
    }

    public function def_assistant_delete() {
        return [
            "description"=> "Delete Assistant",
            "params" => [
                Positional::create("*assistant+|one or more Assistants ID or Name"),
            ]
        ];
    }
    public function cmd_assistant_delete($cmd) {
        $assistantInput = $this->get("assistant");
        foreach($assistantInput as $assistantInput) {
            $assistant = Assistant::fromInput($assistantInput);
            if($assistant) {
                $assistant->delete();
                Output::outLine("Assistant",$assistant->id(),"deleted");
            }
        }
    }
}