<?php
namespace boru\openai\models;

use boru\openai\OpenAI;
use boru\output\Output;
use boru\openai\OpenAIConfig;


class Prompt extends Base {

    /** @var int */
    private $id;
    /** @var string */
    private $assistantId;
    /** @var string */
    private $name;
    /** @var string */
    private $description;
    /** @var int */
    private $order;
    /** @var string */
    private $prompt;

    private static $tableName;
    public static function tableName($tableName=null) {
        if($tableName) {
            static::$tableName = $tableName;
        }
        if(static::$tableName === null) {
            static::$tableName = OpenAI::table("prompts","boru_openai_prompts");
        }
        return static::$tableName;
    }

    public function __construct($idOrArray=null) {
        if ($idOrArray !== null) {
            if (is_array($idOrArray)) {
                $this->loadFromArray($idOrArray);
            } else if (is_numeric($idOrArray)) {
                $this->loadFromId($idOrArray);
            }
        }
    }
    public function loadFromId($id) {
        $db = OpenAI::db();
        if(!$db) {
            return false;
        }
        $sql = "SELECT * FROM ".static::tableName()." WHERE id=?";
        $result = $db->run($sql,[$id]);
        while($row = $db->next($result)) {
            return $this->loadFromArray($row->asArray());
        }
        return false;
    }
    public function loadFromArray($array) {
        if(isset($array["id"])) {
            $this->id = $array["id"];
        }
        if(isset($array["assistant_id"])) {
            $this->assistantId = $array["assistant_id"];
        }
        if(isset($array["name"])) {
            $this->name = $array["name"];
        }
        if(isset($array["description"])) {
            $this->description = $array["description"];
        }
        if(isset($array["order"])) {
            $this->order = $array["order"];
        }
        if(isset($array["prompt"])) {
            $this->prompt = $array["prompt"];
        }
        return $this;
    }

    public function id() {
        return $this->id;
    }
    public function assistantId($assistantId=null) {
        if ($assistantId !== null) {
            $this->assistantId = $assistantId;
        }
        return $this->assistantId;
    }
    public function name($name=null) {
        if ($name !== null) {
            $this->name = $name;
        }
        return $this->name;
    }
    public function description($description=null) {
        if ($description !== null) {
            $this->description = $description;
        }
        return $this->description;
    }
    public function order($order=null) {
        if ($order !== null) {
            $this->order = $order;
        }
        return $this->order;
    }
    public function prompt($prompt=null) {
        if ($prompt !== null) {
            $this->prompt = $prompt;
        }
        return $this->prompt;
    }

    public function toArray() {
        return [
            "id"=>$this->id,
            "assistant_id"=>$this->assistantId,
            "name"=>$this->name,
            "description"=>$this->description,
            "order"=>$this->order,
            "prompt"=>$this->prompt,
        ];
    }

    public function save() {
        $db = OpenAI::db();
        if(!$db) {
            return false;
        }
        if(!$this->order) {
            $this->order = static::getNextOrder($this->assistantId);
        }
        //$db->printDebug(true);
        if(!$this->id) {   
            $sql = "INSERT INTO ".static::tableName()." (assistant_id,name,description,`order`,prompt) VALUES (?,?,?,?,?)";
            $result = $db->run($sql,[$this->assistantId,$this->name,$this->description,$this->order,$this->prompt]);
            $this->id = $db->lastInsertId();
        } else {
            $sql = "UPDATE ".static::tableName()." SET assistant_id=?,name=?,description=?,`order`=?,prompt=? WHERE id=?";
            $result = $db->run($sql,[$this->assistantId,$this->name,$this->description,$this->order,$this->prompt,$this->id]);
        }
        return $this;
    }

    public static function queryById($id) {
        $db = OpenAI::db();
        if(!$db) {
            return false;
        }
        $sql = "SELECT * FROM ".static::tableName()." WHERE id=?";
        $result = $db->run($sql,[$id]);
        while($row = $db->next($result)) {
            return new static($row);
        }
        return false;
    }
    public static function queryByName($name) {
        $db = OpenAI::db();
        if(!$db) {
            return false;
        }
        $sql = "SELECT * FROM ".static::tableName()." WHERE name=?";
        $result = $db->run($sql,[$name]);
        while($row = $db->next($result)) {
            return new static($row);
        }
        return false;
    }
    public static function queryByAssistantId($assistantId) {
        $db = OpenAI::db();
        if(!$db) {
            return false;
        }
        $sql = "SELECT * FROM ".static::tableName()." WHERE assistant_id=? ORDER BY `order` ASC";
        $result = $db->run($sql,[$assistantId]);
        $prompts = [];
        while($row = $db->next($result)) {
            $prompts[] = new static($row->asArray());
        }
        return $prompts;
    }
    public static function getNextOrder($assistantId) {
        $db = OpenAI::db();
        if(!$db) {
            return false;
        }
        $sql = "SELECT MAX(`order`) as `order` FROM ".static::tableName()." WHERE assistantId=?";
        $result = $db->run($sql,[$assistantId]);
        while($row = $db->next($result)) {
            return $row->order + 1;
        }
        return 0;
    }
    public static function forAssistant($assistant,$name=null) {
        $assistantId = $assistant;
        if(is_object($assistant)) {
            $assistantId = $assistant->id();
        }
        $prompts = static::queryByAssistantId($assistantId);
        if(empty($prompts)) {
            return false;
        }
        if($name) {
            foreach($prompts as $prompt) {
                if($prompt->name() == $name) {
                    return $prompt;
                }
            }
            return false;
        }
        return $prompts;
    }

    public static function migrateFromConfig() {
        $promptConfig = OpenAIConfig::get("prompts",[]);
        if(count($promptConfig) > 0) {
            Output::outLine("Migrating prompts from config to database");
            foreach($promptConfig as $assistantTag => $prompts) {
                Output::outLine("Migrating prompts for assistant $assistantTag");
                $assistant = Assistant::queryByTag($assistantTag);
                if($assistant) {
                    Output::outLine("Migrating prompts for assistant $assistantTag");
                    foreach($prompts as $promptName => $promptData) {
                        $check = Prompt::forAssistant($assistant->id(),$promptName);
                        if(!$check) {
                            $prompt = new Prompt();
                            $prompt->assistantId($assistant->id());
                            $prompt->name($promptName);
                            $prompt->prompt($promptData);
                            $prompt->save();
                            if($prompt->id()) {
                                if(OpenAI::isCLI()) {
                                    Output::outLine("Removing prompt $assistantTag.$promptName from config");
                                }
                                OpenAIConfig::delete("prompts.".$assistantTag.".".$promptName);
                            }
                            $check = Prompt::forAssistant($assistant->id(),$promptName);
                            if($check) {
                                OpenAIConfig::delete("prompts.".$assistantTag.".".$promptName);
                            }
                        }
                        else {
                            if(OpenAI::isCLI()) {
                                Output::outLine("Removing prompt $assistantTag.$promptName from config");
                            }
                            OpenAIConfig::delete("prompts.".$assistantTag.".".$promptName);
                        }
                    }
                    if(empty(OpenAIConfig::get("prompts.".$assistantTag))) {
                        OpenAIConfig::delete("prompts.".$assistantTag);
                        OpenAIConfig::save();
                    }
                }
            }
            if(!empty(OpenAIConfig::get("prompts.summary",[]))) {
                OpenAIConfig::delete("prompts.summary");
                OpenAIConfig::save();
            }
            if(empty(OpenAIConfig::get("prompts"))) {
                OpenAIConfig::delete("prompts");
                OpenAIConfig::save();
            }
            OpenAIConfig::save();
        }
    }
}