<?php
namespace boru\output;

class StdOut {
    const ERASE_SCREEN_AFTER            = "\033[0J";
    const ERASE_LINE_BEFORE             = "\033[1K";
    const ERASE_LINE_AFTER              = "\033[0K";
    const MOVE_UP                       = "\033[1A";

    const CODE_SAVE_CURSOR              ="\033[s";
    const CODE_RESTORE_CURSOR           ="\033[u";
    const CODE_CURSOR_IN_SCROLL_AREA    ="\033[1A";

    /** @var StatusBar */
    private $progressBar;

    private static $eol = PHP_EOL;
    /** @var self */
    private static $instance;


    public function __construct() {

    }

    public function writeLine(...$args) {
        static::line(...$args);
    }
    public function write($content) {
        echo $content;
        $this->drawProgressBar();
    }

    public function drawProgressBar() {
        if(!is_null($this->progressBar)) {
            $this->progressBar->draw();
        }
    }

    /**
     * STATIC METHODS
     */

    /**
     * Initialize (if not already) and return the singleton instance. To replace using the $force input, either an instance of StdOut or boolean true.
     * @param StdOut|true|null $force 
     * @return StdOut 
     */
    public static function init($force=null) {
        if(is_object($force)) {
            static::$instance = $force;
        } elseif(is_null(static::$instance) || $force===true) {
            static::$instance = new self();
        }
        return static::$instance;
    }

    /**
     * Print content to the screen -- does not include a newline automatically.
     * @param string $content 
     * @return void 
     */
    public static function output($content) {
        $instance = static::init();
        $instance->write($content);
    }

    public static function drawProgress() {
        $instance = static::init();
        $instance->drawProgressBar();
    }

    /**
     * Generate a line from one or more inputs, applies a newline automtically
     * then prints the line to StdOut::output()
     * @param mixed $parts 
     * @return void 
     */
    public static function line(...$parts) {
        if(!is_array($parts)) {
            $parts = [$parts];
        }
        foreach($parts as $k=>$part) {
            $parts[$k] = StdOut::argToString($part);
        }
        $content = implode(" ",$parts);
        static::init();
        $content = rtrim($content) . static::$eol;
        static::output($content);
    }

    public static function argToString($arg,$json=false) {
        if(is_array($arg) || is_object($arg)) {
            return $json ? json_encode($arg) : print_r($arg,true);
        } else {
            return $arg;
        }
    }
    /**
     * 
     * @param string|string[] $char a string or array of strings with 'start', 'clear', and/or 'up' as values
     * @return string|void 
     */
    public static function ansiChar($char="start|clear|up|color",$extra=null) {
        if(!is_array($char)) {
            if($char == "start") {
                return "\r";
            }
            if($char == "clear") {
                return "\033[2K";
            }
            if($char == "up") {
                return "\033[1A";
            }
            if($char == "color") {
                return is_null($extra) ? : "\033[0m"; "\033[38;5;".$extra."m";
            }
        } else {
            $chars = [];
            foreach($char as $c) {
                $chars[] = static::ansiChar($c);
            }
            return implode("",$chars);
        }
    }
    public static function ansiReset() {

    }
    public static function ansiColor($color,$bg=false) {
        $colorArray["black"]    = [30,40];
        $colorArray["red"]      = [31,41];
        $colorArray["green"]    = [32,42];
        $colorArray["yellow"]   = [33,43];
        $colorArray["blue"]     = [34,44];
        $colorArray["magenta"]  = [35,45];
        $colorArray["cyan"]     = [36,46];
        $colorArray["white"]    = [37,47];
        $colorArray["default"]  = [39,49];
        $colorArray["reset"]    = [0,0];
        if(isset($colorArray[$color])) {
            $code = $bg? $$colorArray[$color][1] : $colorArray[$color][0];
            return "\033[".$code."m";
        }
        return "";
    }

    /**
     * Getters/Setters
     */

    /**
	 * Get the value of progressBar
     * @return  mixed
     */
    public function getProgressBar() {
        return $this->progressBar;
    }

    /**
     * Set the value of progressBar
     * @param   mixed  $progressBar  
     * @return  self
	 */
    public function setProgressBar($progressBar) {
        $this->progressBar = $progressBar;
        return $this;
    }

    /**
     * 
     * @param StatusBar $progressBar 
     * @return void 
     */
    public static function progressBar($progressBar) {
        $instance = static::init();
        $instance->setProgressBar($progressBar);
    }
    public static function sigint($sigintVal=2) {
        if(!is_null(static::$instance->getProgressBar())) {
            static::$instance->getProgressBar()->stop();
        }
        exit();
    }
    public static $cols;
    public static $lines;
    public static function getLines($offset=0,$refresh=false) {
        if(is_null(static::$lines) || $refresh) {
            static::$lines = exec("tput lines");
        }
        return intval(static::$lines-$offset);
    }
    public static function getCols($offset=0,$refresh=false) {
        if(is_null(static::$cols) || $refresh) {
            static::$cols = exec("tput cols");
        }
        return intval(static::$cols-$offset);
    }
}

if(function_exists('pcntl_signal')) {
    pcntl_signal(SIGINT, function() {
        StdOut::sigint(SIGINT);
    });
}