<?php
namespace boru\output\table;

use boru\output\Output;

class Table {
    private $rows = [];
    private $columnProperties = [];

    private $columnPadding = 1;
    private $columnPad = " ";
    private $columnSeparator = "";
    private $outputObject;

    public function __construct(...$rows) {
        $this->rows = $rows;
    }

    public function replace($search,$replace) {
        foreach($this->rows as $row) {
            $row->replace($search,$replace);
        }
        return $this;
    }

    public function displayLength() {
        $length = 0;
        $display = $this->asArray();
        foreach($display as $line) {
            $lineLength = strlen($line);
            if($lineLength > $length) {
                $length = $lineLength;
            }
        }
        return $length;
    }

    /**
     * Add a row to the table
     * @param mixed $row 
     * @return $this
     */
    public function add($row) {
        if($row instanceof Row) {
            $row->table($this);
            $this->rows[] = $row;
            return $this;
        }
        if(is_array($row)) {
            $row = new Row(...$row);            
        } else {
            $row = new Row($row);
        }
        $row->table($this);
        $this->rows[] = $row;
        return $this;
    }
    /**
     * Get or set the rows of the table
     * @param array|null $rows 
     * @return Row[]|Table 
     */
    public function rows($rows=null) {
        if($rows === null) {
            return $this->rows;
        }
        $this->rows = $rows;
        return $this;
    }
    /**
     * Add a row to the table
     * @param mixed ...$cells 
     * @return Row 
     */
    public function row(...$cells) {
        if(count($cells) === 1 && is_array($cells[0])) {
            $cells = $cells[0];
        }
        if(count($cells) === 1 && $cells[0] instanceof Row) {
            $row = $cells[0];
        } else {
            $row = new Row(...$cells);
        }
        $row->table($this);
        $this->rows[] = $row;
        return $row;
    }

    /**
     * Get or set the padding for the columns
     * @param int|null $padding 
     * @return int|Table 
     */
    public function columnProperties($index,$properties) {
        $this->columnProperties[$index] = $properties;
        return $this;
    }
    /**
     * Get or set the padding for the columns
     * @param int|null $padding 
     * @return int|Table 
     */
    public function padding($padding=null) {
        if($padding === null) {
            return $this->columnPadding;
        }
        $this->columnPadding = $padding;
        return $this;
    }
    /**
     * Get or set the padding for the columns
     * @param int|null $padding 
     * @return int|Table 
     */
    public function pad($pad=null) {
        if($pad === null) {
            return $this->columnPad;
        }
        $this->columnPad = $pad;
        return $this;
    }
    /**
     * Get or set the padding for the columns
     * @param int|null $padding 
     * @return int|Table 
     */
    public function separator($separator=null) {
        if($separator === null) {
            return $this->columnSeparator;
        }
        $this->columnSeparator = $separator;
        return $this;
    }

    /**
     * Get or set the outputObject
     * @param Output|null $outputObject 
     * @return Output|Table 
     */
    public function outputObject($outputObject=null) {
        if($outputObject === null) {
            return $this->outputObject;
        }
        $this->outputObject = $outputObject;
        return $this;
    }
    /**
     * Set a property on a column
     * @param int $index
     * @param string $property
     * @param mixed $value
     * @return Table
     * @throws Exception
     */
    public function setOne($index,$property,$value) {
        if(!isset($this->columnProperties[$index])) {
            $this->columnProperties[$index] = [];
        }
        $this->columnProperties[$index][$property] = $value;
        return $this;
    }
    /**
     * Set a properties on a column, or columns. Each iinput is an array [index,property,value]
     * @param mixed $index 
     * @param mixed $property 
     * @param mixed $value 
     * @return $this 
     * @throws Exception 
     */
    public function set(...$args) {
        if(count($args) === 3 && !is_array($args[0])) {
            $a = $args[0];
            $b = $args[1];
            $c = $args[2];
            $args = [[$a,$b,$c]];
        }
        foreach($args as $arg) {
            $this->setOne(...$arg);
        }
        return $this;
    }
    /**
     * Set the width(s) column(s).. inputs can be an array of [index,width] or a list of widths
     * @param int $index
     * @param int $width
     * @return Table
     */
    public function setWidths(...$args) {
        foreach($args as $k=>$arg) {
            if(is_array($arg)) {
                $this->setOne($arg[0],"width",$arg[1]);
            } else {
                $this->setOne($k,"width",$arg);
            }
        }
        return $this;
    }

    public function columnWidth($index,$width=null) {
        if($width === null) {
            if(!isset($this->columnProperties[$index]) || !isset($this->columnProperties[$index]["width"])) {
                return 0;
            }
            return $this->columnProperties[$index]["width"];
        }
        if(!isset($this->columnProperties[$index])) {
            $this->columnProperties[$index] = [];
        }
        $this->columnProperties[$index]["width"] = $width;
        return $this;
    }

    private function buildWidths() {
        foreach($this->rows as $row) {
            $lengths = $row->lengths();
            foreach($lengths as $index=>$length) {
                if(!isset($this->columnProperties[$index])) {
                    $this->columnProperties[$index] = [];
                }
                if(!isset($this->columnProperties[$index]["width"])) {
                    $this->columnProperties[$index]["width"] = 0;
                }
                if($length > $this->columnProperties[$index]["width"]) {
                    $this->columnProperties[$index]["width"] = $length;
                }
            }
        }
    }
    private function compile() {
        $this->buildWidths();
        $lines = [];
        foreach($this->rows as $row) {
            $row->padding($this->padding());
            $row->pad($this->pad());
            $row->separator($this->separator());
            $lines[] = $row->display();
        }
        return $lines;
    }

    public function toArray() {
        return $this->compile();
    }
    public function asArray() {
        return $this->toArray();
    }
    /**
     * Display the table
     * @param Output|null $output 
     * @return void 
     */
    public function display($output=null) {
        $lines = $this->compile();
        if($output === null) {
            $output = $this->outputObject();
        }
        if($output === null) {
            $output = Output::instance();
        }
        foreach($lines as $line) {
            if($output) {
                $output->line($line);
            } else {
                echo $line.PHP_EOL;
            }
        }
    }
    public function show($output=null) {
        $this->display($output);
    }
    public function done() {
        return $this->display();
    }
    public function end() {
        return $this->display();
    }
    public function endTable() {
        return $this->display();
    }
}