<?php

namespace boru\process\Loop;

use boru\process\WorkerInterface;
use boru\process\Status\Reporter\PipeStatusReporter;
use boru\process\Status\Sink\StatusSinkInterface;
use boru\process\Status\WorkerStatusEvent;

/**
 * LoopProvider that uses pcntl_fork and pcntl_wait to manage workers.
 *
 * This implementation:
 * - spawns N workers,
 * - for each worker, creates a pipe for status messages,
 * - reads JSON status lines from all workers,
 * - aggregates "processed" counts based on reported data.
 *
 * It is intentionally simple and blocking; a more advanced implementation
 * can later use an event loop for non-blocking IO and better UX.
 */
class PcntlLoopProvider implements LoopProviderInterface
{
    /**
     * {@inheritdoc}
     */
    public function runWorkers(callable $workerFactory, $numWorkers, StatusSinkInterface $sink = null)
    {
        $numWorkers = (int)$numWorkers;
        if ($numWorkers < 1) {
            $numWorkers = 1;
        }

        $children = array();
        $pipes    = array(); // pid => read-stream

        for ($i = 0; $i < $numWorkers; $i++) {
            $pair = stream_socket_pair(STREAM_PF_UNIX, STREAM_SOCK_STREAM, STREAM_IPPROTO_IP);
            if ($pair === false) {
                // Could not create a pair; skip this worker
                continue;
            }

            list($parentSock, $childSock) = $pair;

            $pid = pcntl_fork();

            if ($pid === -1) {
                // Fork failed; close sockets and break out
                fclose($parentSock);
                fclose($childSock);
                break;
            }
            if($pid > 0) {
                echo "[manager] Fork created for worker $i with pid $pid" . PHP_EOL;
            }
            if ($pid === 0) {
                // Child process
                fclose($parentSock); // child only needs its end

                $reporter = new PipeStatusReporter($childSock);

                // Worker factory is expected to accept an optional StatusReporter,
                // but for BC we try to call it in a flexible way.
                $worker = $this->createWorkerFromFactory($workerFactory, $reporter);

                if (!$worker instanceof WorkerInterface) {
                    // Fail fast; non-zero exit code.
                    fclose($childSock);
                    exit(1);
                }

                $processed = (int)$worker->run();

                fclose($childSock);

                // Child exit code no longer used for processed count aggregation,
                // but we can still exit 0 to indicate success.
                exit(0);
            }

            // Parent process
            fclose($childSock); // parent only needs its end
            stream_set_blocking($parentSock, true);

            $children[]      = $pid;
            $pipes[$pid]     = $parentSock;
        }

        // In parent: read all status messages and wait for children.
        $totalProcessed = $this->collectStatusAndWait($children, $pipes, $sink);

        return $totalProcessed;
    }

    /**
     * Try to create a WorkerInterface instance from the provided factory.
     *
     * @param callable           $factory
     * @param PipeStatusReporter $reporter
     * @return WorkerInterface|null
     */
    protected function createWorkerFromFactory(callable $factory, PipeStatusReporter $reporter)
    {
        // Support both:
        // - function () { return new Worker(...); }
        // - function (StatusReporterInterface $r) { return new Worker(..., $r, ...); }
        $ref = new \ReflectionFunction($factory);
        $numParams = $ref->getNumberOfParameters();
        
        if ($numParams > 0) {
            $worker = call_user_func($factory, $reporter);
        } else {
            $worker = call_user_func($factory);
        }

        return ($worker instanceof WorkerInterface) ? $worker : null;
    }

    /**
     * Read all status lines from worker pipes and wait for children.
     *
     * @param array $children Array of child PIDs.
     * @param array $pipes    pid => read-stream
     * @return int Aggregate "processed" count based on status messages.
     */
    protected function collectStatusAndWait(array $children, array $pipes, StatusSinkInterface $sink = null)
    {
        $perWorkerProcessed = array();
        foreach ($pipes as $pid => $stream) {
            $perWorkerProcessed[$pid] = 0;
            stream_set_blocking($stream, false);
        }

        $active = $pipes;

        while (!empty($active)) {
            foreach ($active as $pid => $stream) {
                $line = fgets($stream);

                if ($line === false) {
                    if (feof($stream)) {
                        fclose($stream);
                        unset($active[$pid]);
                    }
                    continue;
                }

                $line = trim($line);
                if ($line === '') {
                    continue;
                }

                $msg = json_decode($line, true);
                if (!is_array($msg)) {
                    continue;
                }

                $event = new WorkerStatusEvent($pid, $msg);

                if ($sink) {
                    $sink->handle($event);
                } else {
                    echo '[manager] ' . $line . PHP_EOL;
                }

                if (isset($msg['processed'])) {
                    $current = (int)$msg['processed'];
                    if ($current > $perWorkerProcessed[$pid]) {
                        $perWorkerProcessed[$pid] = $current;
                    }
                }
            }

            // Small sleep to avoid busy loop
            usleep(10000);
        }

        foreach ($children as $pid) {
            $status = 0;
            pcntl_waitpid($pid, $status);
        }

        if ($sink) {
            $sink->finish();
        }

        return array_sum($perWorkerProcessed);
    }

}
