<?php

namespace boru\process\Loop;

use boru\process\WorkerInterface;
use boru\process\Status\Reporter\PipeStatusReporter;
use boru\process\Status\Sink\StatusSinkInterface;
use boru\process\Status\WorkerStatusEvent;
use React\EventLoop\Loop;

/**
 * LoopProvider that uses ReactPHP's global Loop to manage worker pipes.
 *
 * It still uses pcntl_fork for worker processes; React is only used
 * in the manager for non-blocking IO and periodic checks.
 */
class ReactLoopProvider implements LoopProviderInterface
{
    /**
     * {@inheritdoc}
     */
    public function runWorkers(callable $workerFactory, $numWorkers, StatusSinkInterface $sink = null)
    {
        $numWorkers = (int)$numWorkers;
        if ($numWorkers < 1) {
            $numWorkers = 1;
        }

        $children           = array(); // pid list
        $perWorkerProcessed = array(); // pid => max processed seen
        $streams            = array(); // pid => stream resource

        // 1) Fork children and set up pipes
        for ($i = 0; $i < $numWorkers; $i++) {
            $pair = stream_socket_pair(STREAM_PF_UNIX, STREAM_SOCK_STREAM, STREAM_IPPROTO_IP);
            if ($pair === false) {
                continue;
            }

            list($parentSock, $childSock) = $pair;

            $pid = pcntl_fork();

            if ($pid === -1) {
                fclose($parentSock);
                fclose($childSock);
                break;
            }

            if ($pid === 0) {
                // Child process
                fclose($parentSock);

                $reporter = new PipeStatusReporter($childSock);

                $worker = $this->createWorkerFromFactory($workerFactory, $reporter);

                if (!$worker instanceof WorkerInterface) {
                    fclose($childSock);
                    exit(1);
                }

                $processed = (int)$worker->run();

                fclose($childSock);
                exit(0);
            }

            // Parent process
            fclose($childSock);
            stream_set_blocking($parentSock, false);

            $children[]            = $pid;
            $perWorkerProcessed[$pid] = 0;
            $streams[$pid]         = $parentSock;

            $this->attachReadStream($parentSock, $pid, $perWorkerProcessed, $streams, $sink);
        }

        // 2) Periodically check for child exit and loop completion
        Loop::addPeriodicTimer(0.1, function ($timer) use (&$children, &$streams, &$perWorkerProcessed, $sink) {
            // Reap any exited children
            foreach ($children as $index => $pid) {
                $status = 0;
                $res    = pcntl_waitpid($pid, $status, WNOHANG);
                if ($res > 0) {
                    unset($children[$index]);
                }
            }

            // When all children have exited and all streams are closed, stop
            if (empty($children) && empty($streams)) {
                if ($sink) {
                    $sink->finish();
                }
                Loop::cancelTimer($timer);
                Loop::stop();
            }
        });

        // 3) Run the loop (blocks until Loop::stop())
        Loop::run();

        // 4) After loop stops, aggregate processed counts
        $totalProcessed = array_sum($perWorkerProcessed);

        return $totalProcessed;
    }

    /**
     * Attach a read stream handler for a worker pipe using React Loop.
     *
     * @param resource                 $stream
     * @param int                      $pid
     * @param array                    $perWorkerProcessed (by ref)
     * @param StatusSinkInterface|null $sink
     */
    protected function attachReadStream(
        $stream,
        $pid,
        array &$perWorkerProcessed,
        array &$streams,
        StatusSinkInterface $sink = null
    ) {
        // We need a reference to the streams array to remove this stream later.
        // We'll store it in a static registry keyed by resource id.
        $streamId = (int)$stream;

        Loop::addReadStream($stream, function ($stream) use ($pid, &$perWorkerProcessed, &$streams, $sink) {
            $line = fgets($stream);

            if ($line === false) {
                if (feof($stream)) {
                    Loop::removeReadStream($stream);
                    fclose($stream);
                    unset($streams[$pid]); // mark this worker's stream closed
                }
                return;
            }

            $line = trim($line);
            if ($line === '') {
                return;
            }

            $msg = json_decode($line, true);
            if (!is_array($msg)) {
                return;
            }

            $event = new WorkerStatusEvent($pid, $msg);

            if ($sink) {
                $sink->handle($event);
            } else {
                echo '[manager] ' . $line . PHP_EOL;
            }

            if (isset($msg['processed'])) {
                $current = (int)$msg['processed'];
                if ($current > $perWorkerProcessed[$pid]) {
                    $perWorkerProcessed[$pid] = $current;
                }
            }
        });
    }

    /**
     * Try to create a WorkerInterface instance from the provided factory.
     *
     * @param callable           $factory
     * @param PipeStatusReporter $reporter
     * @return WorkerInterface|null
     */
    protected function createWorkerFromFactory(callable $factory, PipeStatusReporter $reporter)
    {
        // Support both 0-arg and 1-arg (StatusReporterInterface) factories
        if (is_array($factory)) {
            // method callable; call without reflection
            $worker = call_user_func($factory, $reporter);
        } else {
            try {
                $ref       = new \ReflectionFunction($factory);
                $numParams = $ref->getNumberOfParameters();
            } catch (\ReflectionException $e) {
                $numParams = 0;
            }

            if ($numParams > 0) {
                $worker = call_user_func($factory, $reporter);
            } else {
                $worker = call_user_func($factory);
            }
        }

        return ($worker instanceof WorkerInterface) ? $worker : null;
    }
}
