<?php

namespace boru\process\Queue;

use boru\process\Loop\LoopProviderInterface;
use boru\process\Loop\ReactLoopProvider;
use boru\process\Status\Sink\StatusSinkInterface;
use boru\process\Status\Sink\NullStatusSink;
use boru\process\Queue\Bootstrap\WorkerEnvironmentBuilderInterface;
use boru\queue\Storage\QueueStorageInterface;

class QueueOrchestrator
{
    /**
     * @var WorkerEnvironmentBuilderInterface
     */
    protected $envBuilder;

    /**
     * @var QueueRunnerConfig
     */
    protected $config;

    /**
     * @var LoopProviderInterface
     */
    protected $loopProvider;

    /**
     * @var StatusSinkInterface|null
     */
    protected $defaultSink;

    public function __construct(
        WorkerEnvironmentBuilderInterface $envBuilder,
        QueueRunnerConfig $config = null,
        LoopProviderInterface $loopProvider = null,
        StatusSinkInterface $defaultSink = null
    ) {
        $this->envBuilder   = $envBuilder;
        $this->config       = $config ?: new QueueRunnerConfig();
        $this->loopProvider = $loopProvider ?: new ReactLoopProvider();
        $this->defaultSink  = $defaultSink;
    }

    /**
     * Enqueue a batch of payloads for a single task, then process them.
     *
     * @param string $taskName
     * @param array  $payloads  List of payloads (arrays) to enqueue.
     * @param StatusSinkInterface|null $sink Optional override sink.
     *
     * @return QueueOrchestrationResult
     */
    public function enqueueAndRun($taskName, array $payloads, StatusSinkInterface $sink = null)
    {
        $result = new QueueOrchestrationResult();

        // 1) Use builder once in parent to get storage for enqueuing
        list($storage, /* $registry */) = $this->envBuilder->build();

        if (!$storage instanceof QueueStorageInterface) {
            throw new \RuntimeException('Environment builder did not return QueueStorageInterface.');
        }

        $queueName = $this->config->queueName;

        // 2) Enqueue all payloads
        foreach ($payloads as $payload) {
            // Encode payload as JSON; you can later make this pluggable
            $jsonPayload = json_encode($payload);
            $storage->enqueue($queueName, $taskName, $jsonPayload);
            $result->enqueued++;
        }

        // 3) Run workers using QueueRunner::withBuilder
        $runner = QueueRunner::withBuilder(
            $this->envBuilder,
            $this->configToOptionsArray(),
            $this->loopProvider,
            $this->defaultSink
        );

        $sinkToUse = $sink ?: $this->defaultSink ?: new NullStatusSink();
        $result->processed = $runner->run($sinkToUse);

        // 4) After processing, get a fresh storage instance to check counts
        list($statusStorage, /* $statusRegistry */) = $this->envBuilder->build();

        $result->queuedCount     = $statusStorage->countQueuedItems($queueName);
        $result->processingCount = $statusStorage->countProcessingItems($queueName);
        $result->doneCount       = $statusStorage->countDoneItems($queueName);
        $result->errorCount      = $statusStorage->countErrorItems($queueName);

        return $result;
    }

    /**
     * Helper to turn QueueRunnerConfig into the options array that QueueRunner expects.
     *
     * @return array
     */
    protected function configToOptionsArray()
    {
        return array(
            'queue_name'      => $this->config->queueName,
            'num_workers'     => $this->config->numWorkers,
            'max_items'       => $this->config->maxItems,
            'idle_sleep_us'   => $this->config->idleSleepUs,
            'stop_when_empty' => $this->config->stopWhenEmpty,
            'max_idle_seconds'=> $this->config->maxIdleSeconds,
            'max_run_time'    => $this->config->maxRunTime,
        );
    }
}
