<?php

namespace boru\process\Status\Sink;

use boru\process\Status\WorkerStatusEvent;
use boru\process\Tty\TtyHelper;

class CliStatusBarSink implements StatusSinkInterface
{
    /**
     * @var TtyHelper
     */
    protected $tty;

    /**
     * @var float
     */
    protected $startedAt;

    /**
     * @var int|null
     */
    protected $totalExpected;

    /**
     * @var array pid => ['processed' => int, 'state' => string]
     */
    protected $perWorker = array();

    /**
     * @var bool
     */
    protected $interactive;

    /**
     * @param int|null   $totalExpected Optional total items expected.
     * @param TtyHelper|null $tty
     */
    public function __construct($totalExpected = null, TtyHelper $tty = null)
    {
        $this->startedAt     = microtime(true);
        $this->totalExpected = ($totalExpected !== null) ? (int)$totalExpected : null;
        $this->tty           = $tty ?: new TtyHelper();
        $this->interactive   = $this->tty->isInteractive();

        if ($this->interactive) {
            // Reserve the last line for the bar by printing a newline,
            // then render an initial empty bar.
            echo "\n";
            $this->renderBar();
        }
    }

    public function handle(WorkerStatusEvent $event)
    {
        if (!isset($this->perWorker[$event->pid])) {
            $this->perWorker[$event->pid] = array(
                'processed' => 0,
                'state'     => 'idle',
            );
        }

        if ($event->processed > $this->perWorker[$event->pid]['processed']) {
            $this->perWorker[$event->pid]['processed'] = $event->processed;
        }

        if ($event->state) {
            $this->perWorker[$event->pid]['state'] = $event->state;
        }

        if ($this->interactive) {
            $this->renderBar();
        } else {
            // Non-TTY fallback: very simple output
            if ($event->state === 'processed') {
                $this->printTaskLine($event);
            }
        }
    }

    public function finish()
    {
        if ($this->interactive) {
            $this->renderBar(true);
            echo "\n"; // move past the bar
        } else {
            // For non-TTY, you may want a summary here; keeping it simple:
            $elapsed = microtime(true) - $this->startedAt;
            $total   = $this->getTotalProcessed();
            echo "== processed=$total elapsed=" . sprintf('%.2f', $elapsed) . "s\n";
        }
    }

    protected function renderBar($final = false)
    {
        if (!$this->interactive) {
            return;
        }

        $elapsed = microtime(true) - $this->startedAt;
        $total   = $this->getTotalProcessed();
        $activeWorkers = $this->countActiveWorkers();

        $percent = 0.0;
        $eta     = null;

        if ($this->totalExpected !== null && $this->totalExpected > 0) {
            $percent = ($total / $this->totalExpected) * 100.0;

            if ($total > 0) {
                $rate      = $total / max($elapsed, 0.000001);
                $remaining = $this->totalExpected - $total;
                $eta       = $remaining / max($rate, 0.000001);
            }
        }

        // Get terminal width and leave a small margin to avoid wrapping
        $termWidth = $this->tty->getWidth();
        $maxWidth  = max(20, $termWidth - 2); // leave 2 chars margin

        // Build stats text (left side)
        $left = sprintf(
            "elapsed=%.1fs | %d/%s items | workers=%d",
            $elapsed,
            $total,
            ($this->totalExpected !== null ? $this->totalExpected : '?'),
            $activeWorkers,
        );

        if ($eta !== null) {
            $left = sprintf("ETA=%.1fs | ", $eta) . $left;
        }

        // Build percent text (right side)
        $right = sprintf("%6.2f%%", $percent);

        // Compute available space for bar: total width minus left, right, and spaces/brackets
        // Layout: [BAR] SP LEFT SP RIGHT
        $minBarWidth = 10;
        $fixedPadding = 4; // '[' and ']' and two spaces around them

        $spaceForBar = $maxWidth - strlen($left) - strlen($right) - $fixedPadding;
        if ($spaceForBar < $minBarWidth) {
            // If we don't have enough room, truncate left text to make space
            $truncateTo = max(0, $maxWidth - strlen($right) - $minBarWidth - $fixedPadding);
            if (strlen($left) > $truncateTo) {
                $left = substr($left, 0, $truncateTo);
            }
            $spaceForBar = $minBarWidth;
        }

        $barWidth = max($minBarWidth, $spaceForBar);

        $fillCount = 0;
        if ($this->totalExpected !== null && $this->totalExpected > 0) {
            $fillCount = (int)round(($total / $this->totalExpected) * $barWidth);
        }
        $fillCount = max(0, min($barWidth, $fillCount));

        $bar = str_repeat('=', $fillCount) . str_repeat(' ', max(0, $barWidth - $fillCount));

        // Compose final line, then truncate defensively to maxWidth
        $line = sprintf('%s %s [%s]', $left,$right, $bar);

        if (strlen($line) > $maxWidth) {
            $line = substr($line, 0, $maxWidth);
        }

        // Move to line start, clear, and print
        $this->tty->moveToLineStart();
        $this->tty->eraseCurrentLine();
        echo $line;
        flush();
    }

    protected function getTotalProcessed()
    {
        $total = 0;
        foreach ($this->perWorker as $info) {
            $total += (int)$info['processed'];
        }
        return $total;
    }

    protected function countActiveWorkers()
    {
        $active = 0;
        foreach ($this->perWorker as $info) {
            if ($info['state'] !== 'stopped') {
                $active++;
            }
        }
        return $active;
    }

    /**
     * Optional: print a line above the bar for each completed task.
     */
    protected function printTaskLine(WorkerStatusEvent $event)
    {
        $task = $event->taskName !== null ? $event->taskName : '-';
        $item = $event->itemId !== null ? $event->itemId : '-';

        echo sprintf(
            "[W%5d] done task=%s item=%s processed=%d\n",
            $event->pid,
            $task,
            $item,
            $event->processed
        );
    }
}
