<?php

namespace boru\process\Queue;

use boru\process\Queue\Bootstrap\WorkerEnvironmentBuilderInterface;
use boru\process\Queue\Bootstrap\MysqlWorkerEnvironmentBuilder; // if you have it
use boru\process\Status\Sink\StatusSinkInterface;
use boru\process\Status\Sink\CliStatusBarSink;
use boru\queue\Storage\QueueStorageInterface;
use boru\queue\Task\TaskRegistry;
use boru\queue\Task\ClosureTask;
use boru\queue\Entity\QueueItem;

class QueueProcessRunner
{
    /**
     * @var WorkerEnvironmentBuilderInterface
     */
    protected $envBuilder;

    /**
     * @var QueueRunnerConfig
     */
    protected $config;

    /**
     * @var StatusSinkInterface|null
     */
    protected $sink;

    /**
     * Should we run even if no items were enqueued from enqueueTask()?
     * @var bool
     */
    protected $runIfNoEnqueued = true;

    /**
     * @var callable|null function (Enqueuer $q, QueueStorageInterface $storage, string $queueName)
     */
    protected $enqueueCallback;

    /**
     * @var callable|null function (QueueItem $item, array $payload)
     */
    protected $workerCallback;

    /**
     * @var string
     */
    protected $taskName = 'runner_task';

    /**
     * @var string
     */
    protected $queueName;

    /**
    * @var callable|null function (QueueOrchestrationResult $result): void
    */
    protected $onComplete;

    /**
     * @var callable|null function (QueueItem $item, array $payload, $result): void
     */
    protected $onDone;

    /**
     * Factory: convenient helper for MySQL-based storage using a builder.
     */
    public static function mysql(array $dbConfig, $tableName, callable $taskRegistryFactory = null)
    {
        $builder = new MysqlWorkerEnvironmentBuilder($dbConfig, $tableName, $taskRegistryFactory);

        return new self($builder);
    }

    public function __construct(WorkerEnvironmentBuilderInterface $envBuilder)
    {
        $this->envBuilder = $envBuilder;
        $this->config     = QueueRunnerConfig::fromArray([
            'stop_when_empty' => true,
            'max_idle_seconds'=> 0.3,
        ]);
        $this->queueName  = $this->generateQueueName();
    }

    protected function generateQueueName()
    {
        return 'queue_' . date('Ymd_His') . '_' . substr(sha1(uniqid('', true)), 0, 8);
    }

    public function queueName($queueName)
    {
        $this->queueName = (string)$queueName;
        return $this;
    }

    public function numWorkers($numWorkers)
    {
        $this->config->numWorkers = (int)$numWorkers;
        return $this;
    }

    public function stopWhenEmpty($stop = true)
    {
        $this->config->stopWhenEmpty = (bool)$stop;
        return $this;
    }

    public function maxItemsPerWorker($max)
    {
        $this->config->maxItems = (int)$max;
        return $this;
    }

    public function maxIdleSeconds($seconds=1.0)
    {
        $this->config->maxIdleSeconds = (float)$seconds;
        return $this;
    }

    public function runIfNoEnqueued($run = true)
    {
        $this->runIfNoEnqueued = (bool)$run;
        return $this;
    }

    public function withSink(StatusSinkInterface $sink)
    {
        $this->sink = $sink;
        return $this;
    }

    /**
     * Register the enqueue closure. callbacks can get up to 3 methods.. only the first one is generally needed to enqueue with $q->enqueue(array [payload]).
     *
     * @param callable $callback function (Enqueuer $q, QueueStorageInterface $storage, string $queueName): void
     * @return $this
     */
    public function enqueueTask(callable $callback)
    {
        $this->enqueueCallback = $callback;
        return $this;
    }

    /**
     * Register the worker closure.
     *
     * @param callable $callback function (QueueItem $item, array $payload): mixed
     * @return $this
     */
    public function workerTask(callable $callback)
    {
        $this->workerCallback = $callback;
        return $this;
    }

    /**
     * Register a callback to be called once after the run completes.
     *
     * @param callable $callback function (QueueOrchestrationResult $result): void
     * @return $this
     */
    public function onComplete(callable $callback)
    {
        $this->onComplete = $callback;
        return $this;
    }

    /**
     * Register a callback called after each successful task.
     *
     * @param callable $callback function (QueueItem $item, array $payload, $result): void
     * @return $this
     */
    public function onDone(callable $callback)
    {
        $this->onDone = $callback;
        return $this;
    }

    /**
     * Run the full pipeline: enqueue via child, then process with workers.
     *
     * @return QueueOrchestrationResult
     */
    public function run()
    {
        if (!$this->enqueueCallback) {
            throw new \RuntimeException('enqueueTask() must be set before run()');
        }

        if (!$this->workerCallback) {
            throw new \RuntimeException('workerTask() must be set before run()');
        }

        // Build a taskRegistryFactory that registers our worker callback as a ClosureTask
        $taskName = $this->taskName;
        $workerCb = $this->workerCallback;
        $onDone   = $this->onDone;

        $taskRegistryFactory = function () use ($taskName, $workerCb, $onDone) {
            $registry = new TaskRegistry();
            $wrapped = function (QueueItem $item, array $payload) use ($workerCb, $onDone) {
                $result = call_user_func($workerCb, $item, $payload);

                if ($onDone) {
                    call_user_func($onDone, $item, $payload, $result);
                }

                return $result;
            };

            $registry->register(new ClosureTask($taskName, $wrapped));
            return $registry;
        };

        // Wrap existing envBuilder to inject our TaskRegistry
        $wrappedBuilder = new class($this->envBuilder, $taskRegistryFactory) implements WorkerEnvironmentBuilderInterface {
            private $inner;
            private $factory;

            public function __construct(WorkerEnvironmentBuilderInterface $inner, callable $factory)
            {
                $this->inner   = $inner;
                $this->factory = $factory;
            }

            public function build()
            {
                list($storage, /* $existingRegistry */) = $this->inner->build();

                if (!$storage instanceof QueueStorageInterface) {
                    throw new \RuntimeException('Environment builder did not return QueueStorageInterface');
                }

                $registry = call_user_func($this->factory);

                return array($storage, $registry);
            }
        };

        // Use QueueOrchestrator with our wrapped builder
        $config = clone $this->config;
        $config->queueName = $this->queueName;
        $config->stopWhenEmpty = $this->config->stopWhenEmpty;
        $config->maxIdleSeconds = $this->config->maxIdleSeconds;

        // Use totalExpected for bar, if sink is CliStatusBarSink you can pass count later; for now leave null
        $orchestrator = new QueueOrchestrator(
            $wrappedBuilder,
            $config,
            null,
            $this->sink
        );

        // Enqueue & run using the same "enqueue child + workers" pattern.
        // Instead of passing payloads, we delegate enqueue work to the callback
        // inside the child, then just run QueueRunner.
        $result = $this->orchestratedEnqueueAndRun($orchestrator, $wrappedBuilder, $taskName);

        if ($this->onComplete) {
            call_user_func($this->onComplete, $result);
        }

        return $result;
    }

    /**
     * Internal: mimic QueueOrchestrator::enqueueAndRun but delegate enqueue logic
     * to the user-provided enqueue callback.
     *
     * @param QueueOrchestrator              $orchestrator
     * @param WorkerEnvironmentBuilderInterface $builder
     * @param string                         $taskName
     *
     * @return QueueOrchestrationResult
     */
    protected function orchestratedEnqueueAndRun(
        QueueOrchestrator $orchestrator,
        WorkerEnvironmentBuilderInterface $builder,
        $taskName
    ) {
        $queueName = $this->queueName;

        $enqueueCb = $this->enqueueCallback;

        // In parent before fork:
        $pair = stream_socket_pair(STREAM_PF_UNIX, STREAM_SOCK_STREAM, STREAM_IPPROTO_IP);
        list($parentSock, $childSock) = $pair;

        // 1) Fork enqueue child
        $pid = pcntl_fork();

        if ($pid === -1) {
            throw new \RuntimeException('Could not fork enqueue process');
        }

        if ($pid === 0) {
            // CHILD: enqueue
            fclose($parentSock);
            try {
                list($storage, /* $registry */) = $builder->build();
                if (!$storage instanceof QueueStorageInterface) {
                    fwrite(STDERR, "enqueue child: builder did not return QueueStorageInterface\n");
                    exit(1);
                }
                $enqueuer = new Enqueuer($storage, $queueName, $taskName);
                call_user_func($enqueueCb, $enqueuer, $storage, $queueName);

                $enqueuedCount = $enqueuer->getEnqueuedCount();
                $summary = [
                    'enqueued_count' => $enqueuedCount,
                ];

                $result    = new QueueOrchestrationResult();
                $result->enqueued      = $storage->countQueuedItems($queueName);
                $result->queuedCount   = $result->enqueued;
                $result->processingCount = $storage->countProcessingItems($queueName);
                $result->doneCount     = $storage->countDoneItems($queueName);
                $result->errorCount    = $storage->countErrorItems($queueName);

                $data = json_encode($result->toArray());
                fwrite($childSock, $data);
                fclose($childSock);

                exit(0);
            } catch (\Exception $e) {
                fwrite(STDERR, "enqueue child exception: " . $e->getMessage() . "\n");
                exit(1);
            }
        }
        
        // PARENT: wait for enqueue child
        $status = 0;
        pcntl_waitpid($pid, $status);

        if (!pcntl_wifexited($status) || pcntl_wexitstatus($status) !== 0) {
            throw new \RuntimeException(
                'Enqueue child failed (exit code ' . pcntl_wexitstatus($status) . ')'
            );
        }

        fclose($childSock);
        // Read enqueue summary from child
        $data = '';
        while (!feof($parentSock)) {
            $data .= fread($parentSock, 8192);
        }
        fclose($parentSock);

        // build result.
        $result = new QueueOrchestrationResult();
        if ($data) {
            $summary = json_decode($data, true);
            if (isset($summary['enqueued'])) {
                // We have a valid summary from enqueue child, let's use it so we can
                // display accurate counts right away.
                $result = QueueOrchestrationResult::fromArray($summary);
            }
        }

        if($result->enqueued === 0 && $result->queuedCount === 0 && !$this->runIfNoEnqueued) {
            // No items enqueued, and we don't want to run workers.
            if($this->sink) {
                $this->sink->finish();
            }
            return $result;
        }

        // 2) Run workers via QueueRunner in orchestrator
        // reuse the orchestrator's enqueueAndRun for now:
        // since we already enqueued, we just need "run" behavior.
        // You can add a dedicated runOnly() in QueueOrchestrator later;
        // for now, call QueueRunner directly.

        $runner = QueueRunner::withBuilder(
            $builder,
            $orchestrator->configToOptionsArray(), // make configToOptionsArray public or expose config
            null,
            $this->sink
        );

        $sinkToUse       = $this->sink ?: new CliStatusBarSink($result->enqueued);
        if($sinkToUse instanceof CliStatusBarSink) {
            $sinkToUse->setExpected($result->enqueued);
        }
        $result->processed = $runner->run($sinkToUse);

        // 3) Update final counts
        list($finalStorage, /* $finalRegistry */) = $builder->build();
        if ($finalStorage instanceof QueueStorageInterface) {
            $result->queuedCount     = $finalStorage->countQueuedItems($queueName);
            $result->processingCount = $finalStorage->countProcessingItems($queueName);
            $result->doneCount       = $finalStorage->countDoneItems($queueName);
            $result->errorCount      = $finalStorage->countErrorItems($queueName);
        }
        /*if($this->sink) {
            $this->sink->finish();
        }*/
        return $result;
    }
}
