<?php

namespace boru\process\Queue;

use boru\dhutils\dhGlobal;
use boru\process\ProcessManager;
use boru\process\Loop\LoopProviderInterface;
use boru\process\Loop\ReactLoopProvider;
use boru\process\Queue\Bootstrap\WorkerEnvironmentBuilderInterface;
use boru\process\Queue\Worker\QueueWorker;
use boru\process\Queue\Worker\WorkerRuntimeOptions;
use boru\process\Status\Reporter\NullStatusReporter;
use boru\process\Status\Sink\NullStatusSink;
use boru\process\Status\Reporter\StatusReporterInterface;
use boru\process\Status\Sink\StatusSinkInterface;
use boru\queue\Storage\QueueStorageInterface;
use boru\queue\Task\TaskRegistry;

/**
 * High-level runner for processing a queue with multiple workers.
 *
 * This is the primary public entrypoint for queue-based workloads.
 */
class QueueRunner
{
    /**
     * @var QueueRunnerConfig
     */
    protected $config;

    /**
     * @var QueueStorageInterface
     */
    protected $storage;

    /**
     * @var string
     */
    protected $queueName;

    /**
     * @var TaskRegistry
     */
    protected $taskRegistry;

    /**
     * @var array
     */
    protected $managerOptions = array();

    /**
     * @var WorkerRuntimeOptions
     */
    protected $workerOptions;

    /**
     * @var LoopProviderInterface
     */
    protected $loopProvider;

    /**
     * @var WorkerEnvironmentBuilderInterface|null
     */
    protected $envBuilder;

    /**
     * @var StatusSinkInterface|null
     */
    protected $defaultSink;

    /**
     * Core constructor; prefer using named constructors queueWithStorage() or withBuilder().
     * @param QueueStorageInterface  $storage
     * @param string                 $queueName
     * @param TaskRegistry           $taskRegistry
     * @param array                  $options
     *        - num_workers     int   number of worker processes
     *        - max_items       int   per-worker max items (0 = unlimited)
     *        - idle_sleep_us   int   microseconds sleep when idle
     *        - stop_when_empty bool  stop worker after being idle long enough (future)
     * @param LoopProviderInterface  $loopProvider
     */
    public function __construct(
        QueueRunnerConfig $config,
        LoopProviderInterface $loopProvider = null,
        QueueStorageInterface $storage = null,
        TaskRegistry $taskRegistry = null,
        WorkerEnvironmentBuilderInterface $envBuilder = null,
        StatusSinkInterface $defaultSink = null
    ) {
        $this->config       = $config;
        $this->loopProvider = $loopProvider ?: new ReactLoopProvider();
        $this->storage      = $storage;
        $this->taskRegistry = $taskRegistry;
        $this->envBuilder   = $envBuilder;
        $this->defaultSink  = $defaultSink;
    }

    /**
     * Simple mode: you already have storage + registry.
     */
    public static function queueWithStorage(
        QueueStorageInterface $storage,
        TaskRegistry $taskRegistry,
        array $options = array(),
        LoopProviderInterface $loopProvider = null,
        StatusSinkInterface $defaultSink = null
    ) {
        $config = QueueRunnerConfig::fromArray($options);

        return new self($config, $loopProvider, $storage, $taskRegistry, null, $defaultSink);
    }

    /**
     * Builder mode: construct storage+registry per worker.
     */
    public static function withBuilder(
        WorkerEnvironmentBuilderInterface $envBuilder,
        array $options = array(),
        LoopProviderInterface $loopProvider = null,
        StatusSinkInterface $defaultSink = null
    ) {
        $config = QueueRunnerConfig::fromArray($options);

        return new self($config, $loopProvider, null, null, $envBuilder, $defaultSink);
    }

    /**
     * Run the queue using the configured workers and loop provider.
     *
     * @param StatusSinkInterface|null $sink Optional sink for status/progress output.
     * @return int Total number of items processed by all workers (best effort).
     */
    public function run(StatusSinkInterface $sink = null)
    {

        $sinkToUse = $sink ?: ($this->defaultSink ?: new NullStatusSink());

        $manager = new ProcessManager($this->loopProvider, array(
            'num_workers' => $this->config->numWorkers,
        ));

        $queueName    = $this->config->queueName;
        $envBuilder   = $this->envBuilder;
        $storage      = $this->storage;
        $taskRegistry = $this->taskRegistry;

        // Prepare worker runtime options DTO
        $workerRuntimeOptions = new WorkerRuntimeOptions(array(
            'max_items'       => $this->config->maxItems,
            'idle_sleep_us'   => $this->config->idleSleepUs,
            'stop_when_empty' => $this->config->stopWhenEmpty,
            'max_idle_seconds'=> $this->config->maxIdleSeconds,
            'max_run_time'    => $this->config->maxRunTime,
        ));

        $workerFactory = function (StatusReporterInterface $reporter = null) use (
            $storage,
            $taskRegistry,
            $envBuilder,
            $queueName,
            $workerRuntimeOptions
        ) {
            $statusReporter = $reporter ?: new NullStatusReporter();

            if ($envBuilder instanceof WorkerEnvironmentBuilderInterface) {
                list($workerStorage, $workerRegistry) = $envBuilder->build();
            } else {
                $workerStorage  = $storage;
                $workerRegistry = $taskRegistry;
            }

            return new QueueWorker(
                $workerStorage,
                $queueName,
                $workerRegistry,
                $statusReporter,
                $workerRuntimeOptions
            );
        };

        return $manager->run($workerFactory, $sinkToUse);
    }
}
