<?php

namespace boru\process\Queue\Worker;

/**
 * Encapsulates per-worker runtime limits and stop behavior.
 */
class WorkerRuntimeOptions
{
    public $idleSleepUs;
    public $maxItems;
    public $stopWhenEmpty;
    public $maxIdleSeconds;
    public $maxRunTime;

    public function __construct(array $options = array())
    {
        $this->idleSleepUs   = isset($options['idle_sleep_us']) ? (int)$options['idle_sleep_us'] : 100000;
        $this->maxItems      = isset($options['max_items']) ? (int)$options['max_items'] : 0;
        $this->stopWhenEmpty = !empty($options['stop_when_empty']);
        $this->maxIdleSeconds= isset($options['max_idle_seconds']) ? (float)$options['max_idle_seconds'] : 0.0;
        $this->maxRunTime    = isset($options['max_run_time']) ? (float)$options['max_run_time'] : 0.0;
    }

    /**
     * Decide if the worker should stop based on counters and timers.
     *
     * @param int        $processed    Items processed so far.
     * @param float      $startedAt    Timestamp when worker started (microtime(true)).
     * @param float|null $idleStarted  Timestamp when idle streak began, or null if not idle.
     *
     * @return array [bool $shouldStop, string|null $reason]
     */
    public function shouldStop($processed, $startedAt, $idleStarted)
    {
        $now = microtime(true);

        // Hard runtime cap
        if ($this->maxRunTime > 0) {
            $elapsed = $now - $startedAt;
            if ($elapsed >= $this->maxRunTime) {
                return array(true, 'max_run_time');
            }
        }

        // Per-worker item cap
        if ($this->maxItems > 0 && $processed >= $this->maxItems) {
            return array(true, 'max_items');
        }

        // Stop when empty: idle-based timeout
        if ($this->stopWhenEmpty && $this->maxIdleSeconds > 0 && $idleStarted !== null) {
            $idleElapsed = $now - $idleStarted;
            if ($idleElapsed >= $this->maxIdleSeconds) {
                return array(true, 'max_idle_seconds');
            }
        }

        return array(false, null);
    }
}
