<?php

namespace boru\process\Status\Sink;

use boru\process\Status\WorkerStatusEvent;

/**
 * Simple CLI sink that prints per-event lines and a final summary.
 *
 * This is intentionally minimal: easy to read and easy to extend later
 * into a full status bar or progress display.
 */
class CliStatusSink implements StatusSinkInterface
{
    /**
     * @var float
     */
    protected $startedAt;

    /**
     * @var array pid => max processed seen
     */
    protected $perWorkerProcessed = array();

    public function __construct()
    {
        $this->startedAt = microtime(true);
    }

    public function handle(WorkerStatusEvent $event)
    {
        if (!isset($this->perWorkerProcessed[$event->pid])) {
            $this->perWorkerProcessed[$event->pid] = 0;
        }

        // Track max processed for this worker
        if ($event->processed > $this->perWorkerProcessed[$event->pid]) {
            $this->perWorkerProcessed[$event->pid] = $event->processed;
        }

        // Per-event output
        switch ($event->state) {
            case 'busy':
                $this->printLine($event, 'busy');
                break;

            case 'processed':
                $this->printLine($event, 'done ');
                break;

            case 'error':
                $this->printError($event);
                break;

            case 'stopped':
                $this->printStopped($event);
                break;

            case 'idle':
                // Probably too noisy to print every idle; leave silent for now
                break;

            case 'debug':
                $this->printDebug($event);
                break;
        }
    }

    public function finish()
    {
        $elapsed = microtime(true) - $this->startedAt;

        $totalProcessed = 0;
        foreach ($this->perWorkerProcessed as $pid => $count) {
            $totalProcessed += (int)$count;
        }

        $numWorkers = count($this->perWorkerProcessed);

        echo sprintf(
            "== workers=%d processed=%d elapsed=%.2fs\n",
            $numWorkers,
            $totalProcessed,
            $elapsed
        );
    }

    /**
     * Print a normal per-task line (busy/done).
     *
     * @param WorkerStatusEvent $event
     */
    protected function printLine(WorkerStatusEvent $event, $label)
    {
        $task  = $event->taskName !== null ? $event->taskName : '-';
        $item  = $event->itemId !== null ? $event->itemId : '-';

        echo sprintf(
            "[W%5d] %-7s task=%-15s item=%-8s processed=%d\n",
            $event->pid,
            $label,
            $task,
            $item,
            $event->processed
        );
    }

    /**
     * Print an error event.
     *
     * @param WorkerStatusEvent $event
     */
    protected function printError(WorkerStatusEvent $event)
    {
        $task = $event->taskName !== null ? $event->taskName : '-';
        $item = $event->itemId !== null ? $event->itemId : '-';
        $msg  = ($event->error && isset($event->error['message']))
            ? $event->error['message']
            : 'unknown error';

        echo sprintf(
            "[W%5d] error   task=%-15s item=%-8s msg=%s\n",
            $event->pid,
            $task,
            $item,
            $msg
        );
    }

    /**
     * Print a stopped event.
     *
     * @param WorkerStatusEvent $event
     */
    protected function printStopped(WorkerStatusEvent $event)
    {
        echo sprintf(
            "[W%5d] stopped processed=%d\n",
            $event->pid,
            $event->processed
        );
    }

    /**
     * Print a debug event.
     *
     * @param WorkerStatusEvent $event
     */
    protected function printDebug(WorkerStatusEvent $event)
    {
        // For now, use 'taskName' or error['message'] as the debug message if present
        $msg = '';
        if ($event->taskName !== null) {
            $msg = $event->taskName;
        } elseif ($event->error && isset($event->error['message'])) {
            $msg = $event->error['message'];
        }

        echo sprintf(
            "[W%5d] debug   %s\n",
            $event->pid,
            $msg
        );
    }
}
