<?php

namespace boru\process\Queue;

use boru\process\Loop\LoopProviderInterface;
use boru\process\Loop\ReactLoopProvider;
use boru\process\Status\Sink\StatusSinkInterface;
use boru\process\Status\Sink\NullStatusSink;
use boru\process\Queue\Bootstrap\WorkerEnvironmentBuilderInterface;
use boru\queue\Storage\QueueStorageInterface;

class QueueOrchestrator
{
    /**
     * @var WorkerEnvironmentBuilderInterface
     */
    protected $envBuilder;

    /**
     * @var QueueRunnerConfig
     */
    protected $config;

    /**
     * @var LoopProviderInterface
     */
    protected $loopProvider;

    /**
     * @var StatusSinkInterface|null
     */
    protected $defaultSink;

    public function __construct(
        WorkerEnvironmentBuilderInterface $envBuilder,
        QueueRunnerConfig $config = null,
        LoopProviderInterface $loopProvider = null,
        StatusSinkInterface $defaultSink = null
    ) {
        $this->envBuilder   = $envBuilder;
        $this->config       = $config ?: new QueueRunnerConfig();
        $this->loopProvider = $loopProvider ?: new ReactLoopProvider();
        $this->defaultSink  = $defaultSink;
    }

    /**
     * Enqueue a batch of payloads in a dedicated child process, then
     * run a multi-process QueueRunner to process them.
     *
     * @param string                 $taskName
     * @param array                  $payloads  List of payloads (arrays/scalars) to enqueue.
     * @param StatusSinkInterface|null $sink    Optional override sink.
     *
     * @return QueueOrchestrationResult
     */
    public function enqueueAndRun($taskName, array $payloads, StatusSinkInterface $sink = null)
    {
        $result    = new QueueOrchestrationResult();
        $queueName = $this->config->queueName;

        // 1) Fork an enqueue child process
        $pid = pcntl_fork();

        if ($pid === -1) {
            throw new \RuntimeException('Could not fork enqueue process');
        }

        if ($pid === 0) {
            // CHILD: enqueue work in its own process and DB env.
            try {
                list($storage, /* $registry */) = $this->envBuilder->build();

                if (!$storage instanceof QueueStorageInterface) {
                    // Fail fast if builder isn't configured correctly
                    fwrite(STDERR, "enqueue child: envBuilder did not return QueueStorageInterface\n");
                    exit(1);
                }

                foreach ($payloads as $payload) {
                    $jsonPayload = is_string($payload) ? $payload : json_encode($payload);
                    $storage->enqueue($queueName, $taskName, $jsonPayload);
                }

                exit(0);
            } catch (\Exception $e) {
                // Basic error reporting; you can improve logging later.
                fwrite(STDERR, "enqueue child exception: " . $e->getMessage() . "\n");
                exit(1);
            }
        }

        // PARENT: wait for enqueue child to finish
        $status = 0;
        pcntl_waitpid($pid, $status);

        if (!pcntl_wifexited($status) || pcntl_wexitstatus($status) !== 0) {
            throw new \RuntimeException(
                'Enqueue child failed (exit code ' . pcntl_wexitstatus($status) . ')'
            );
        }

        // All payloads should now be enqueued by the child.
        $result->enqueued = count($payloads);

        // 2) Run workers using QueueRunner with the builder
        $runner = QueueRunner::withBuilder(
            $this->envBuilder,
            $this->configToOptionsArray(),
            $this->loopProvider,
            $this->defaultSink
        );

        $sinkToUse       = $sink ?: $this->defaultSink ?: new NullStatusSink();
        $result->processed = $runner->run($sinkToUse);

        // 3) Snapshot queue state using a fresh env in the parent
        list($statusStorage, /* $statusRegistry */) = $this->envBuilder->build();

        if ($statusStorage instanceof QueueStorageInterface) {
            $result->queuedCount     = $statusStorage->countQueuedItems($queueName);
            $result->processingCount = $statusStorage->countProcessingItems($queueName);
            $result->doneCount       = $statusStorage->countDoneItems($queueName);
            $result->errorCount      = $statusStorage->countErrorItems($queueName);
        }

        return $result;
    }

    /**
     * Helper: turn QueueRunnerConfig into an options array for QueueRunner.
     *
     * @return array
     */
    public function configToOptionsArray()
    {
        return array(
            'queue_name'      => $this->config->queueName,
            'num_workers'     => $this->config->numWorkers,
            'max_items'       => $this->config->maxItems,
            'idle_sleep_us'   => $this->config->idleSleepUs,
            'stop_when_empty' => $this->config->stopWhenEmpty,
            'max_idle_seconds'=> $this->config->maxIdleSeconds,
            'max_run_time'    => $this->config->maxRunTime,
        );
    }
}
