<?php

namespace boru\process\Queue\Worker;

use boru\process\WorkerInterface;
use boru\process\Status\Reporter\StatusReporterInterface;
use boru\process\Status\Reporter\NullStatusReporter;
use boru\queue\Storage\QueueStorageInterface;
use boru\queue\Task\TaskRegistry;
use boru\queue\Entity\QueueItem;
use boru\process\Queue\Worker\WorkerRuntimeOptions;

/**
 * A blocking worker that pulls items from a boru\queue storage
 * and executes their associated tasks.
 *
 * Designed to run inside a single PHP process without an event loop.
 * A separate ProcessManager will be responsible for forking and IPC.
 */
class QueueWorker implements WorkerInterface
{
    /**
     * @var QueueStorageInterface
     */
    protected $storage;

    /**
     * @var string
     */
    protected $queueName;

    /**
     * @var TaskRegistry
     */
    protected $taskRegistry;

    /**
     * @var StatusReporterInterface
     */
    protected $statusReporter;

    /**
     * @var WorkerRuntimeOptions
     */
    protected $options = [];

    /**
     * @var bool
     */
    protected $running = true;

    /**
     * @param QueueStorageInterface     $storage
     * @param string                    $queueName
     * @param TaskRegistry              $taskRegistry
     * @param StatusReporterInterface   $statusReporter
     * @param WorkerRuntimeOptions      $options
     */
    public function __construct(
        QueueStorageInterface $storage,
        $queueName,
        TaskRegistry $taskRegistry,
        StatusReporterInterface $statusReporter = null,
        WorkerRuntimeOptions $options = null
    ) {
        $this->storage        = $storage;
        $this->queueName      = $queueName;
        $this->taskRegistry   = $taskRegistry;
        $this->statusReporter = $statusReporter ?: new NullStatusReporter();
        $this->options        = $options;
    }

    /**
     * Run the worker loop until completion.
     *
     * @return int Number of processed items.
     */
    public function run()
    {
        
        $this->debug("worker options = " . json_encode($this->options));
        $processed = 0;
        $startedAt = microtime(true);
        $idleStarted = null;

        while ($this->running) {
            list($shouldStop, $reason) = $this->options->shouldStop($processed, $startedAt, $idleStarted);
            if ($shouldStop) {
                $this->debug('stopping', array('reason' => $reason, 'processed' => $processed));
                break;
            }

            $item = $this->reserveNextItem();
            
            if (!$item) {
                $this->reportStatus('idle', $processed);

                if ($idleStarted === null) {
                    $idleStarted = microtime(true);
                }

                if ($this->options->idleSleepUs > 0) {
                    usleep($this->options->idleSleepUs);
                }
                continue;
            }

            // We got work; reset idle streak
            $idleStarted = null;

            $this->reportStatus('busy', $processed, $item);

            try {
                $this->processItem($item);
                $processed++;
                $this->reportStatus('processed', $processed, $item);
            } catch (\Exception $e) {
                $this->handleProcessingError($item, $e);
                $this->reportStatus('error', $processed, $item, $e);
            }
        }

        $this->reportStatus('stopped', $processed);

        return $processed;
    }

    /**
     * External hooks (e.g. from signals) can call this to stop the loop
     * gracefully at the next opportunity.
     *
     * @return void
     */
    public function stop()
    {
        $this->running = false;
    }

    /**
     * Reserve the next item from the queue for processing.
     *
     * @return QueueItem|null
     */
    protected function reserveNextItem()
    {
        return $this->storage->reserveNext($this->queueName);
    }

    /**
     * Execute the task associated with the given queue item.
     *
     * @param QueueItem $item
     * @return void
     */
    protected function processItem(QueueItem $item)
    {
        $taskName = $item->getTaskName();

        // payload is stored as a string in storage; for now assume JSON.
        $rawPayload = $item->getPayload();
        $payload    = $this->decodePayload($rawPayload);

        $task = $this->taskRegistry->get($taskName);
        if (!$task) {
            throw new \RuntimeException('No task registered for name: ' . $taskName);
        }

        // We assume the task contract matches TaskRegistry expectations:
        // e.g. $task->execute(QueueItem $item, array $payload)
        $result = $task->execute($item, $payload);

        // Mark the item as done with a JSON-encoded result (or null).
        $encodedResult = $this->encodeResult($result);
        $this->storage->markDone($item, $encodedResult);
    }

    /**
     * Handle a processing error. Marks the item as error in the storage.
     *
     * @param QueueItem   $item
     * @param \Exception  $e
     * @return void
     */
    protected function handleProcessingError(QueueItem $item, \Exception $e)
    {
        // Store a simple JSON-encoded error payload; can be enriched later.
        $errorPayload = json_encode([
            'message' => $e->getMessage(),
            'code'    => $e->getCode(),
        ]);

        $this->storage->markError($item, $errorPayload);
    }

    /**
     * Helper to send structured status updates.
     *
     * @param string          $state
     * @param int             $processed
     * @param QueueItem|null  $item
     * @param \Exception|null $e
     * @return void
     */
    protected function reportStatus($state, $processed, QueueItem $item = null, \Exception $e = null)
    {
        $data = [
            'state'     => $state,
            'processed' => $processed,
            'queue'     => $this->queueName,
            'time'      => microtime(true),
        ];

        if ($item) {
            $data['item_id']   = $item->getId();
            $data['task_name'] = $item->getTaskName();
        }

        if ($e) {
            $data['error'] = [
                'message' => $e->getMessage(),
                'code'    => $e->getCode(),
            ];
        }

        $this->statusReporter->report($data);
    }

    protected function debug($message, array $extra = array())
    {
        $data = array_merge(
            array(
                'state'     => 'debug',
                'queue'     => $this->queueName,
                'time'      => microtime(true),
                'message'   => $message,
            ),
            $extra
        );

        $this->statusReporter->report($data);
    }


    /**
     * Decode payload stored in the queue item. For now assume JSON, but
     * this can be customized or made pluggable later.
     *
     * @param string|null $raw
     * @return array
     */
    protected function decodePayload($raw)
    {
        if ($raw === null || $raw === '') {
            return [];
        }
        if(is_array($raw)){
            return $raw;
        }
        $decoded = json_decode($raw, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            // If it's not valid JSON, wrap it as-is.
            return ['raw' => $raw];
        }

        return $decoded;
    }

    /**
     * Encode task result for storage. For now JSON-encode if not null.
     *
     * @param mixed $result
     * @return string|null
     */
    protected function encodeResult($result)
    {
        if ($result === null) {
            return null;
        }

        // If it's already a string, we can store directly, but using JSON
        // consistently keeps round-tripping simple.
        return json_encode($result);
    }
}
