<?php

namespace boru\process\Tty;

class TtyHelper
{
    const ERASE_SCREEN_AFTER         = "\033[0J";
    const ERASE_LINE_BEFORE          = "\033[1K";
    const ERASE_LINE_AFTER           = "\033[0K";

    const CODE_SAVE_CURSOR           = "\033[s";
    const CODE_RESTORE_CURSOR        = "\033[u";

    /**
     * Check if STDOUT looks like a TTY.
     *
     * @return bool
     */
    public function isInteractive()
    {
        if (function_exists('stream_isatty')) {
            return @stream_isatty(STDOUT);
        }

        if (function_exists('posix_isatty')) {
            return @posix_isatty(STDOUT);
        }

        return (PHP_SAPI === 'cli');
    }

    /**
     * Get terminal width in characters.
     *
     * @return int
     */
    public function getWidth()
    {
        // Try tput cols if available
        $width = 0;

        if (PHP_SAPI === 'cli') {
            $cmd = 'tput cols 2>/dev/null';
            $output = @shell_exec($cmd);
            if ($output !== null) {
                $w = (int)trim($output);
                if ($w > 0) {
                    $width = $w;
                }
            }
        }

        if ($width <= 0 && function_exists('shell_exec')) {
            // Fallback: environment variable
            $cols = getenv('COLUMNS');
            if ($cols !== false) {
                $width = (int)$cols;
            }
        }

        if ($width <= 0) {
            $width = 80; // sane default
        }

        return $width;
    }

    public function eraseCurrentLine()
    {
        echo self::ERASE_LINE_AFTER;
    }

    public function saveCursor()
    {
        echo self::CODE_SAVE_CURSOR;
    }

    public function restoreCursor()
    {
        echo self::CODE_RESTORE_CURSOR;
    }

    /**
     * Move cursor to the beginning of the current line.
     */
    public function moveToLineStart()
    {
        echo "\r";
    }

    /**
     * Move cursor up N lines.
     *
     * @param int $lines
     */
    public function moveCursorUp($lines = 1)
    {
        if ($lines < 1) {
            return;
        }
        echo "\033[" . (int)$lines . "A";
    }

    /**
     * Move cursor down N lines.
     *
     * @param int $lines
     */
    public function moveCursorDown($lines = 1)
    {
        if ($lines < 1) {
            return;
        }
        echo "\033[" . (int)$lines . "B";
    }

    public function setScrollRegion($top, $bottom)
    {
        echo "\033[" . (int)$top . ";" . (int)$bottom . "r";
    }

    public function resetScrollRegion()
    {
        // 0;0 = full screen scroll region
        echo "\033[0;0r";
    }

    /**
     * Get terminal height (rows).
     *
     * @return int
     */
    public function getHeight()
    {
        $height = 0;

        if (PHP_SAPI === 'cli') {
            $output = @shell_exec('tput lines 2>/dev/null');
            if ($output !== null) {
                $h = (int)trim($output);
                if ($h > 0) {
                    $height = $h;
                }
            }
        }

        if ($height <= 0) {
            $height = 24; // reasonable default
        }

        return $height;
    }

}
