<?php
namespace boru\qdrant;

use boru\dot\Dot;
use boru\output\Output;
use boru\qdrant\response\QdrantResponse;
use Exception;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\RequestException;

class Qdrant {
    private $config = [
        "host"=>"localhost",
        "port"=>6333,
        "apikey" => false,
        "docker"=>[
            "image"=>"qdrant/qdrant",
            "name"=>"qdrant",
            "apiPort"=>6333,
            "grpcPort"=>6334,
            "dataPath"=>__DIR__."/../data",
        ]
    ];

    private static $debugMode = false;
    private static $debugHttp = false;

    /**
     * @var this
     */
    private static $instance;

    private static $guzzleOptions = [
        'timeout' => 60, // Response timeout
        'connect_timeout' => 5, // Connection timeout
    ];

    public function __construct($config=[]) {
        if(!empty($config)) {
            $this->config($config);
        }
        if(!static::$instance) {
            static::$instance = $this;
        }
    }

    public static function instance($config=[]) {
        if(!static::$instance) {
            static::$instance = new static($config);
        }
        return static::$instance;
    }

    public function config($config=[]) {
        if(!empty($config)) {
            $this->config = array_merge($this->config,$config);
        }
        return $this->config;
    }
    public function getConfig($key=null,$default=null) {
        return Dot::get($this->config,$key,$default);
    }
    public function setConfig($key,$value) {
        return Dot::set($this->config,$key,$value);
    }
    public function getEndpoint($path=null) {
        $host = $this->getConfig("host");
        $port = $this->getConfig("port");
        $url = "http://$host:$port";
        if($path) {
            if(substr($path,0,1) !== "/") {
                $path = "/$path";
            }
            $url .= $path;
        }
        return $url;
    }

    public function guzzleHeaders($contentType="application/json") {
        $headers = [
            //"Authorization"=>"Bearer " . Config::get("openai.api_key"),
            "Accept"=>"application/json",
            "User-Agent"=>"OpenAI-PHP/0.1",
            "OpenAI-Source"=>"boru/openai-php",
            "OpenAI-Beta"=>"assistants=v2"
        ];
        return $headers;
    }

    public function guzzleOptions(...$options) {
        $return = static::$guzzleOptions;
        foreach($options as $option) {
            if(is_array($option)) {
                $return = array_merge($return,$option);
            }
        }
        return $return;
    }

    /**
     * @param mixed $type 
     * @param string $path 
     * @param array $parameters 
     * @return ApiBaseResponse|false 
     * @throws Exception 
     */
    public function request($type,$path="",$parameters=[],$guzzleOptions=[]) {
        $type = strtoupper($type);
        $url = $this->getEndpoint($path);
        self::printHttpDebug(strtoupper($type),"Requesting to: ".$url);
        $options = [];
        $options["headers"] = $this->guzzleHeaders();
        if(!empty($parameters) && is_array($parameters)) {
            $options['json'] = $parameters;
        }
        $options = $this->guzzleOptions($options,$guzzleOptions);
        $client = new \GuzzleHttp\Client();
        $response = static::callGuzzle($client,strtoupper($type),$url,$options);
        $parsed = static::responseParse($response);
        return QdrantResponse::fromArray($parsed);
    }

    private static function responseParse($response) {
        $body = $response->getBody()->getContents();
        $json = json_decode($body,true);
        if($json === null) {
            return $body;
        }
        return $json;
    }
    /**
     * @param \GuzzleHttp\Client $client 
     * @param string $type 
     * @param string $url 
     * @param array $options 
     * @return \Psr\Http\Message\ResponseInterface 
     * @throws Exception 
     */
    private static function callGuzzle($client,$type,$url,$options) {
        $i=0;
        $errorMessage = "";
        while($i++<3) {
            try {
                $response = $client->request(strtoupper($type),$url,$options);
                self::printHttpDebug(strtoupper($type),"Completed");
                return $response;
            } catch(RequestException $e) {
                $errorMessage = $e->getResponse()->getBody()->getContents();
                self::printHttpDebug(strtoupper($type),"Error with request.. retrying..");
            } catch(ClientException $e) {
                $errorMessage = $e->getResponse()->getBody()->getContents();
                self::printHttpDebug(strtoupper($type),"Error with request.. retrying..");
            } catch (Exception $e) {
                $errorMessage = $e->getMessage();
                self::printHttpDebug(strtoupper($type),"Error with request.. retrying..");
            }
        }
        if(empty($errorMessage)) {
            $errorMessage = $e->getMessage();
        }
        throw new Exception($errorMessage);
    }

    public static function debug($debug=null) {
        if($debug !== null) {
            static::$debugMode = $debug ? true : false;
        }
        return static::$debugMode ? true : false;
    }
    public static function debugHttp($debug=null) {
        if($debug !== null) {
            static::$debugHttp = $debug ? true : false;
        }
        return static::$debugHttp ? true : false;
    }

    public static function printDebug(...$messages) {
        if(static::debug()) {
            array_unshift($messages,"[DEBUG]");
            Output::outLine(...$messages);
        }
    }
    public static function printHttpDebug(...$messages) {
        if(static::debugHttp()) {
            array_unshift($messages,"[HTTP]");
            Output::outLine(...$messages);
        }
    }
}