<?php
namespace boru\qdrant\Models;

use boru\qdrant\Api\PointsAPI;
use boru\qdrant\Qdrant;
use boru\qdrant\Response\QdrantResponse;

class Point extends Base {
    private $id;
    private $score; //if a search result
    private $version; //if a search result
    private $distance; //if a search result
    private $vector;
    private $payload = [];

    private $embeddingCallback = null;

    public function __construct($id,$vector,$payload=[]) {
        $this->id = $id;
        $this->vector = $vector;
        $this->payload = $payload;
    }

    public function toArray() {
        return [
            "id" => (int) $this->id,
            "vector" => $this->vector,
            "payload" => $this->payload
        ];
    }

    public function id($id=null) {
        if($id !== null) {
            $this->id = $id;
        }
        return $this->id;
    }
    public function vector($vector=null) {
        if($vector !== null) {
            $this->vector = $vector;
        }
        return $this->vector;
    }
    public function payload($payload=null) {
        if($payload !== null) {
            $this->payload = $payload;
        }
        return $this->payload;
    }
    public function score($score=null) {
        if($score !== null) {
            $this->score = $score;
        }
        return $this->score;
    }
    public function version($version=null) {
        if($version !== null) {
            $this->version = $version;
        }
        return $this->version;
    }
    public function distance($distance=null) {
        if($distance !== null) {
            $this->distance = $distance;
        }
        return $this->distance;
    }

    /**
     * Set's the callback function to generate the vector for the point. It takes one argument, the point itself. Must return a vector.
     * @param mixed $callback 
     * @return callable|false 
     */
    public function embedder($callback=null) {
        if($callback !== null) {
            $this->embeddingCallback = $callback;
        }
        return $this->embeddingCallback;
    }

    public function embed($data) {
        if($this->embeddingCallback !== null) {
            return call_user_func($this->embeddingCallback,$data);
        }
        return false;
    }

    public function upsert($collectionName) {
        $api = new PointsAPI($this->client());
        if(empty($this->vector())) {
            $this->vector = $this->embed($this);
        }
        if($this->vector === false) {
            throw new \Exception("No vector found for point");
        }
        return $api->upsert($collectionName,["points"=>[$this->toArray()]]);
    }

    public static function fromId($collectionName,$id,$client=null) {
        if($client === null) {
            $client = Qdrant::instance();
        }
        $api = new PointsAPI($client);
        $response = $api->get($collectionName,["ids"=>$id]);
        return static::fromArray($response);
    }

    public static function fromArray($data) {
        if(!isset($data["id"]) || (!isset($data["vector"]) && !isset($data["payload"]))) {
            throw new \Exception("Invalid Point data\n\n".json_encode($data,JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE));
        }
        if(!isset($data["payload"])) {
            $data["payload"] = [];
        }
        if(!isset($data["vector"])) {
            $data["vector"] = [];
        }
        if(!is_array($data["vector"])) {
            $data["vector"] = json_decode($data["vector"],true);
        }
        if(!is_array($data["payload"])) {
            $data["payload"] = json_decode($data["payload"],true);
        }
        return new Point($data["id"],$data["vector"],$data["payload"]);
    }

    /**
     * 
     * @param array|QdrantResponse $response 
     * @return void 
     */
    public static function fromResponse($response) {
        if($response instanceof QdrantResponse) {
            $array = $response->get("result");
        } else {
            $array = $response;
        }
        if(!is_array($array)) {
            throw new \Exception("Invalid Point data");
        }
        if(isset($array["id"])) {
            return static::fromArray($array);
        }
        $points = [];
        foreach($array as $point) {
            $points[] = static::fromArray($point);
        }
        return $points;
    }
}