<?php
namespace boru\qdrant\Filter;

/**
 * Builds a Qdrant-compatible filter payload.
 */
class Filter
{
    /** @var array  */
    private $must    = [];
    /** @var array  */
    private $should  = [];
    /** @var array  */
    private $mustNot = [];

    /**
     * Start adding `must` conditions.
     * This is used to require certain conditions in the results.
     * @return FilterGroup
     */
    public function must()
    {
        return new FilterGroup($this, 'must');
    }

    /**
     * Start adding `should` conditions.
     * This is used to boost certain conditions in the results.
     * @return FilterGroup
     */
    public function should()
    {
        return new FilterGroup($this, 'should');
    }

    /**
     * Start adding `must_not` conditions.
     * This is used to exclude certain conditions from the results.
     * @return FilterGroup
     */
    public function mustNot()
    {
        return new FilterGroup($this, 'must_not');
    }

    /**
     * Internal: add a compiled condition to the given clause.
     * @param string $clause The clause to which the condition belongs (must, should, must_not).
     * @param array $condition The condition to add.
     * @return void
     */
    public function add($clause, $condition)
    {
        switch ($clause) {
            case 'must':
                $this->must[] = $condition;
                break;
            case 'should':
                $this->should[] = $condition;
                break;
            case 'must_not':
                $this->mustNot[] = $condition;
                break;
        }
    }

    /**
     * Export the complete filter array for Qdrant.
     * @return array
     */
    public function toArray()
    {
        $filter = [];

        if (!empty($this->must)) {
            $filter['must'] = $this->must;
        }

        if (!empty($this->should)) {
            $filter['should'] = $this->should;
        }

        if (!empty($this->mustNot)) {
            $filter['must_not'] = $this->mustNot;
        }

        return ['filter' => $filter];
    }
}