<?php
namespace boru\qdrant\Filter;

/**
 * Builds a Qdrant-compatible filter payload.
 */
class Filter
{
    /** @var array  */
    private $must    = [];
    /** @var array  */
    private $should  = [];
    /** @var array  */
    private $minShould  = [];
    /** @var array  */
    private $mustNot = [];

    /**
     * Start adding `must` conditions.
     * This is used to require certain conditions in the results.
     * @return FilterGroup
     */
    public function must()
    {
        $group = new FilterGroup($this, 'must');
        $this->must[] = $group;
        return $group;
    }

    /**
     * Start adding `should` conditions.
     * This is used to boost certain conditions in the results.
     * @return FilterGroup
     */
    public function should()
    {
        $group = new FilterGroup($this, 'should');
        $this->should[] = $group;
        return $group;
    }

    /**
     * Start adding `should` conditions.
     * This is used to boost certain conditions in the results.
     * @return FilterGroup
     */
    public function minShould()
    {
        $group = new FilterGroup($this, 'minShould');
        $this->minShould[] = $group;
        return $group;
    }

    /**
     * Start adding `must_not` conditions.
     * This is used to exclude certain conditions from the results.
     * @return FilterGroup
     */
    public function mustNot()
    {
        $group = new FilterGroup($this, 'must_not');
        $this->mustNot[] = $group;
        return $group;
    }

    /**
     * Internal: add a compiled condition to the given clause.
     * @param string $clause The clause to which the condition belongs (must, should, must_not).
     * @param array $condition The condition to add.
     * @return void
     */
    public function add($clause, $condition)
    {
        switch ($clause) {
            case 'must':
                $this->must[] = $condition;
                break;
            case 'should':
                $this->should[] = $condition;
                break;
            case 'must_not':
                $this->mustNot[] = $condition;
                break;
        }
    }

    /**
     * Export the complete filter array for Qdrant.
     * @return array
     */
    public function toArray()
    {
        $filter = [];

        if (!empty($this->must)) {
            foreach($this->must as $group) {
                $filter[] = $group->toArray();
            }
            //$filter['must'] = $this->must;
        }

        if (!empty($this->should)) {
            foreach($this->should as $group) {
                $filter[] = $group->toArray();
            }
            //$filter['should'] = $this->should;
        }

        if (!empty($this->mustNot)) {
            foreach($this->mustNot as $group) {
                $filter[] = $group->toArray();
            }
            //$filter['must_not'] = $this->mustNot;
        }
        if (!empty($this->minShould)) {
            foreach($this->minShould as $group) {
                $filter[] = $group->toArray();
            }
            //$filter['min_should'] = $this->minShould;
        }
        return ['filter' => $filter];
    }
}