<?php
namespace boru\qdrant\Models;

use boru\dot\Dot;
use boru\qdrant\Api\CollectionsAPI;
use boru\qdrant\Api\PointsAPI;
use boru\qdrant\Api\SearchAPI;
use boru\qdrant\Qdrant;

class Collection extends Base {
    private $name;
    private $options = [
        "on_disk_payload" => null,
        "hnsw_config" => [],
        "wal_config" => [],
        "optimizers_config"=> [],
        "quantization_config" => [],
        "sparse_vectors" => [],
        "vectors" => [],
    ];
    
    public function __construct($name,$vector=null,$options=[]) {
        $this->name = $name;
        if($vector !== null) {
            $this->options["vectors"] = $vector;
        }
        if(is_array($options)) {
            foreach($options as $key => $value) {
                $this->set($key,$value);
            }
        }
    }

    public function toArray($forConfig=false) {
        if(!$forConfig) {
            return [
                "name" => $this->name,
                "options" => $this->options
            ];
        }
        $array = $this->options;
        foreach($array as $key => $value) {
            if($value === null || empty($value)) {
                unset($array[$key]);
            } elseif(is_object($value)) {
                $array[$key] = $value->toArray();
            }
        }
        return $array;
    }

    public function api($client=null) {
        if($client === null) {
            $client = Qdrant::instance();
        }
        return new CollectionsAPI($client);
    }
    public function pointsApi($client=null) {
        if($client === null) {
            $client = Qdrant::instance();
        }
        return new PointsAPI($client);
    }

    public function name($name=null) {
        if($name !== null) {
            $this->name = $name;
        }
        return $this->name;
    }
    public function options($options=null) {
        if($options !== null) {
            $this->options = $options;
        }
        return $this->options;
    }
    public function set($key,$value) {
        return Dot::set($this->options,$key,$value);
    }
    public function get($key,$default=null) {
        return Dot::get($this->options,$key,$default);
    }

    public function delete($client=null,$verify=false) {
        if($verify !== true) {
            if($client instanceof Point || is_array($client) || is_string($client)) {
                // If the client is a point, string id, or an array of points, we will delete the points instead of the collection
                return $this->deletePoint($client);
            }
            throw new \Exception("Delete collection is not verified. This is specifically to prevent accidental deletion of collections when attempting to delete points.");
        }
        if($client === null) {
            $client = Qdrant::instance();
        }
        $api = new CollectionsAPI($client);
        return $api->delete($this->name);
    }
    public function create($client=null) {
        return $this->save($client);
    }
    public function save($client=null) {
        if(empty($this->name)) {
            throw new \Exception("Collection name is required");
        }
        if($client === null) {
            $client = Qdrant::instance();
        }
        $api = new CollectionsAPI($client);
        if($api->exists($this->name)) {
            //return $api->update($this->name,$this->toArray(true));
        } else {
            return $api->create($this->name,$this->toArray(true));
        }
    }

    public function getCollection($client=null) {
        if($client === null) {
            $client = Qdrant::instance();
        }
        $api = new CollectionsAPI($client);
        $response = $api->get($this->name);
        return $response;
    }

    public function addPoint($point,$wait=true,$client=null) {
        return $this->addPoints([$point],$wait,$client);
    }

    public function addPoints($points=[],$wait=true,$client=null) {
        if($client === null) {
            $client = Qdrant::instance();
        }
        $api = new PointsAPI($client);
        foreach($points as $point) {
            if($point instanceof Point) {
                $point->collection($this);
            }
        }
        $response = $api->upsert($this->name,["points"=>$points],$wait);
        if($wait) {
            if($response->get("result.status") !== "completed") {
                throw new \Exception("Failed to add points to collection: ".$response->get("result.status")." - ".$response->get());
            }
            return true;
        } else {
            return $response;
        }
    }
    public function getPoint($id,$client=null) {
        if($client === null) {
            $client = Qdrant::instance();
        }
        $api = new PointsAPI($client);
        $response = $api->get($this->name,$id);
        return Point::fromResponse($response);
    }
    public function deletePoint($idOrPoint,$client=null) {
        if($client === null) {
            $client = Qdrant::instance();
        }
        if($idOrPoint instanceof Point) {
            $id = $idOrPoint->id();
        } else {
            $id = $idOrPoint;
        }
        $api = new PointsAPI($client);
        $response = $api->delete($this->name,$id);
        return $response;
    }

    public function searchRaw($options=[],$consistency=null,$timeout=null) {
        $api = new SearchAPI(Qdrant::instance());
        return $api->search($this->name,$options,$consistency,$timeout);
    }
    public function search($vector=null,$filter=null,$limit=5,$withPayload=true,$withVector=false) {
        $options = [];
        if($vector !== null && !empty($vector)) {
            $options["vector"] = $vector;
        }
        if($filter !== null && !empty($filter)) {
            $options["filter"] = $filter;
        }
        $options["limit"] = $limit;
        if($withPayload) {
            $options["with_payload"] = $withPayload;
        }
        if($withVector) {
            $options["with_vector"] = $withVector;
        }
        $response = $this->searchRaw($options);
        if(is_array($response->get("result"))) {
            $results = new Points([],$this);
            foreach($response->get("result") as $result) {
                $results[] = Point::fromArray($result);
            }
            return $results;
        }
        return false;
    }
}