<?php
namespace boru\qdrant\Models;

use boru\qdrant\Api\PointsAPI;
use boru\qdrant\Qdrant;
use boru\qdrant\Response\QdrantResponse;
use Exception;

class Point extends Base {
    private $id;
    private $score; //if a search result
    private $version; //if a search result
    private $distance; //if a search result
    private $vector;
    private $payload = [];
    /** @var Collection */
    private $collection = null;

    public function __construct($id=null,$vector=[],$payload=[]) {
        if($id === null) {
            $id = static::makeUuid();
        }
        $this->id = $id;
        $this->vector = $vector;
        $this->payload = $payload;
    }

    public function toArray() {
        if(is_numeric($this->id)) {
            $id = (int) $this->id;
        } else {
            $id = $this->id;
        }
        return [
            "id" => $id,
            "vector" => $this->vector,
            "payload" => $this->payload
        ];
    }

    public function id($id=null) {
        if($id !== null) {
            $this->id = $id;
        }
        if($this->id === null) {
            $this->id = static::makeUuid();
        }
        if(is_numeric($this->id)) {
            return (int) $this->id;
        }
        if(is_string($this->id)) {
            return (string) $this->id;
        }
        return $this->id;
    }
    public function vector($vector=null) {
        if($vector !== null) {
            $this->vector = $vector;
        }
        return $this->vector;
    }
    public function payload($payload=null) {
        if($payload !== null) {
            $this->payload = $payload;
        }
        return $this->payload;
    }
    public function score($score=null) {
        if($score !== null) {
            $this->score = $score;
        }
        return $this->score;
    }
    public function version($version=null) {
        if($version !== null) {
            $this->version = $version;
        }
        return $this->version;
    }
    public function distance($distance=null) {
        if($distance !== null) {
            $this->distance = $distance;
        }
        return $this->distance;
    }

    public function collection($collection=null) {
        if($collection !== null) {
            if($collection instanceof Collection) {
                $this->collection = $collection;
            } else {
                $this->collection = new Collection($collection);
            }
            $this->collection = $collection;
        }
        return $this->collection;
    }
    public function collectionName() {
        if($this->collection === null || !$this->collection instanceof Collection) {
            return false;
        }
        return $this->collection->name();
    }

    /**
     * 
     * @param Collection|string|null $collection 
     * @return boru\qdrant\ApiBaseResponse|false 
     * @throws Exception 
     */
    public function upsert($collection=null) {
        $collectionName = null;
        if(is_string($collection)) {
            $collection = new Collection($collection);
            $collectionName = $collection->name();
        }
        if($collection === null) {
            $collection = $this->collection();
            $collectionName = $collection->name();
        }
        if($collection === null) {
            throw new \Exception("No collection found for point");
        }
        $api = new PointsAPI($this->client());
        if(!$this->vector()) {
            throw new \Exception("No vector found for point");
        }
        return $api->upsert($collectionName,["points"=>[$this->toArray()]]);
    }

    public static function fromId($collectionName,$id,$client=null) {
        if($client === null) {
            $client = Qdrant::instance();
        }
        $api = new PointsAPI($client);
        $response = $api->get($collectionName,["ids"=>$id]);
        return static::fromArray($response);
    }

    public static function fromArray($data) {
        if(!isset($data["id"]) || (!isset($data["vector"]) && !isset($data["payload"]))) {
            throw new \Exception("Invalid Point data\n\n".json_encode($data,JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE));
        }
        if(!isset($data["payload"])) {
            $data["payload"] = [];
        }
        if(!isset($data["vector"])) {
            $data["vector"] = [];
        }
        if(!is_array($data["vector"])) {
            $data["vector"] = json_decode($data["vector"],true);
        }
        if(!is_array($data["payload"])) {
            $data["payload"] = json_decode($data["payload"],true);
        }
        $point = new Point($data["id"],$data["vector"],$data["payload"]);
        if(isset($data["score"])) {
            $point->score($data["score"]);
        }
        if(isset($data["version"])) {
            $point->version($data["version"]);
        }
        if(isset($data["distance"])) {
            $point->distance($data["distance"]);
        }
        return $point;
    }

    /**
     * 
     * @param array|QdrantResponse $response 
     * @return void 
     */
    public static function fromResponse($response) {
        if($response instanceof QdrantResponse) {
            $array = $response->get("result");
        } else {
            $array = $response;
        }
        if(!is_array($array)) {
            throw new \Exception("Invalid Point data");
        }
        if(isset($array["id"])) {
            return static::fromArray($array);
        }
        $points = [];
        foreach($array as $point) {
            $points[] = static::fromArray($point);
        }
        return $points;
    }

    public static function makeUuid() {
        $uid = \Ramsey\Uuid\Uuid::uuid4();
        return $uid->toString();
    }
}