<?php
namespace boru\query\models;

use boru\query\models\interfaces\SelectInterface;
use boru\query\models\Table;
use boru\dhutils\traits\JsonTrait;

class Column implements SelectInterface {
    use JsonTrait;
    private $name = "";
    private $alias = "";
    private $type = "";
    private $collation = "";
    private $null = true;
    private $key;
    private $default;
    private $extra;

    private $isPrimary = false;

    private $table;

    public function __construct($options=[],$table=null) {
        if($options) {
            $this->fromArray($options);
        }
        $this->table($table);
    }

    public function fromArray($array) {
        foreach($array as $key => $value) {
            $keyName = strtolower($key);
            if($keyName == "field") {
                $keyName = "name";
            } elseif($keyName == "null") {
                if($value == "NO") {
                    $value = false;
                } else {
                    $value = true;
                }
            }
            if(property_exists($this, $keyName)) {
                $this->$keyName = $value;
            }
        }
        return $this;
    }

    public function toArray() {
        return [
            "name" => $this->name,
            "type" => $this->type,
            "collation" => $this->collation,
            "null" => $this->null,
            "key" => $this->key,
            "default" => $this->default,
            "extra" => $this->extra
        ];
    }

    public function toSqlSelect() {
        if(!$this->table) {
            throw new \Exception("Column must have a table");
        }
        $selectName = $this->table->toSql().".`".$this->name."`";
        if($this->alias) {
            $selectName .= " AS `".$this->alias."`";
        }
        return $selectName;
    }

    public function toSql($input=null) {
        if(!$this->table) {
            throw new \Exception("Column must have a table");
        }
        if($this->alias) {
            return "`".$this->alias."`";
        }
        return $this->table->toSql().".`".$this->name."`";
    }

    /**
     * 
     * @param mixed $name 
     * @return string 
     */
    public function name($name=null) {
        if($name !== null) {
            $this->name = $name;
        }
        return $this->name;
    }

    public function alias($alias=null) {
        if($alias !== null) {
            $this->alias = $alias;
        }
        return $this->alias;
    }

    /**
     * 
     * @return string 
     */
    public function refName() {
        if($this->alias) {
            return $this->alias;
        }
        return $this->name;
    }

    public function tableRef() {
        return $this->table->refName();
    }

    /**
     * 
     * @param mixed $type 
     * @return mixed 
     */
    public function type($type=null) {
        if($type !== null) {
            $this->type = $type;
        }
        return $this->type;
    }

    /**
     * 
     * @param mixed $collation 
     * @return mixed 
     */
    public function collation($collation=null) {
        if($collation !== null) {
            $this->collation = $collation;
        }
        return $this->collation;
    }

    /**
     * 
     * @param mixed $null 
     * @return bool 
     */
    public function null($null=null) {
        if($null !== null) {
            $this->null = $null ? true : false;
        }
        return $this->null ? true : false;
    }

    /**
     * 
     * @param mixed $default 
     * @return mixed 
     */
    public function default($default=null) {
        if($default !== null) {
            $this->default = $default;
        }
        if(!$this->null && empty($this->default)) {
            if($this->type == "int") {
                return 0;
            }
            if($this->type == "float") {
                return 0.0;
            }
            if($this->type == "bool") {
                return false;
            }
            return "";
        }
        return $this->default;
    }

    /**
     * 
     * @param mixed $key 
     * @return mixed 
     */
    public function key($key=null) {
        if($key !== null) {
            $this->key = $key;
        }
        return $this->key;
    }

    /**
     * 
     * @param mixed $default 
     * @return mixed 
     */
    public function extra($extra=null) {
        if($extra !== null) {
            $this->extra = $extra;
        }
        return $this->extra;
    }

    public function isPrimary($isPrimary=null) {
        if($isPrimary !== null) {
            $this->isPrimary = $isPrimary;
        }
        return $this->isPrimary;
    }

    /**
     * 
     * @param string|Table $table 
     * @return Table 
     */
    public function table($table=null) {
        if($table !== null) {
            if(!$table instanceof Table) {
                if(is_string($table)) {
                    $table = new Table($table);
                } else {
                    throw new \Exception("Table must be a string or an instance of Table");
                }
            }
            $this->table = $table;
        }
        return $this->table;
    }

    public function typeFix(&$value) {
        if($value instanceof Value) {
            $value->value($this->typeFixValue($value->value()));
        } else {
            $value = $this->typeFixValue($value);
        }
    }

    private function typeFixValue($value) {
        if($this->null && empty($value)) {
            $value = null;
        } else {
            if(empty($value)) {
                $value = $this->default();
            }
            if($this->type == "int") {
                $value = (int)$value;
            }
            if($this->type == "float") {
                $value = (float)$value;
            }
            if($this->type == "bool") {
                $value = (bool)$value;
            }
        }
        return $value;
    }
}